package ru.yandex.solomon.expression.ast.serialization;

import java.util.HashMap;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.IntNode;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.Ast;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
class AstMappers {
    private AstMappers() {
    }

    private static final HashMap<String, AstMapper> MAPPER_BY_TYPE = new HashMap<>();
    private static final HashMap<Class<? extends Ast>, AstMapper> MAPPER_BY_CLASS = new HashMap<>();

    static final String TYPE_KEY = "_t";
    static final String RANGE_KEY = "_r";

    static {
        register(new AstAnonymousMapper());
        register(new AstAssignmentMapper());
        register(new AstBinOpMapper());
        register(new AstCallMapper());
        register(new AstIdentMapper());
        register(new AstInterpolatedStringMapper());
        register(new AstLambdaMapper());
        register(new AstOpMapper());
        register(new AstSelectorMapper());
        register(new AstSelectorsMapper());
        register(new AstTernaryOpMapper());
        register(new AstUnaryOpMapper());
        register(new AstUseMapper());
        register(new AstValueDoubleMapper());
        register(new AstValueStringMapper());
        register(new AstValueDurationMapper());
    }

    private static <T extends Ast> void register(AstMapper<T> astMapper) {
        if (MAPPER_BY_TYPE.containsKey(astMapper.getType())) {
            throw new RuntimeException("Duplicated mapper for type: " + astMapper.getType());
        }
        MAPPER_BY_TYPE.put(astMapper.getType(), astMapper);
        MAPPER_BY_CLASS.put(astMapper.getAstClass(), astMapper);
    }

    static <T extends Ast> AstMapper<T> mapperByType(String type) {
        @SuppressWarnings("unchecked")
        AstMapper<T> mapper = MAPPER_BY_TYPE.get(type);
        if (mapper == null) {
            throw new IllegalArgumentException("Ast type is not supported: " + type);
        }
        return mapper;
    }

    static <T extends Ast> AstMapper<T> mapperByClass(Class<? extends Ast> astClass) {
        @SuppressWarnings("unchecked")
        AstMapper<T> mapper = MAPPER_BY_CLASS.get(astClass);
        if (mapper == null) {
            throw new IllegalArgumentException("Ast type is not supported: " + astClass.getSimpleName());
        }
        return mapper;
    }

    static List<IntNode> encodeRange(PositionRange range) {
        return List.of(
                IntNode.valueOf(range.getBegin().getLine()),
                IntNode.valueOf(range.getBegin().getColumn()),
                IntNode.valueOf(range.getBegin().getOffset()),
                IntNode.valueOf(range.getEnd().getLine()),
                IntNode.valueOf(range.getEnd().getColumn()),
                IntNode.valueOf(range.getEnd().getOffset())
        );
    }

    static PositionRange decodeRange(@Nullable JsonNode json) {
        if (json == null) {
            return PositionRange.UNKNOWN;
        }
        var iterator = json.elements();
        int begLine = iterator.next().asInt();
        int begColumn = iterator.next().asInt();
        int begOffset = iterator.next().asInt();
        int endLine = iterator.next().asInt();
        int endColumn = iterator.next().asInt();
        int endOffset = iterator.next().asInt();
        return PositionRange.of(begLine, begColumn, begOffset, endLine, endColumn, endOffset);
    }

}
