package ru.yandex.solomon.expression.ast.serialization;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.Ast;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class AstMappingContext {
    private static final ObjectMapper mapper = new ObjectMapper();

    private final boolean includeRanges;

    public AstMappingContext(boolean includeRanges) {
        this.includeRanges = includeRanges;
    }

    public <T extends Ast> T parse(JsonNode json) {
        String type = json.get(AstMappers.TYPE_KEY).textValue();
        AstMapper<T> parser = AstMappers.mapperByType(type);
        PositionRange range;
        if (includeRanges) {
            range = AstMappers.decodeRange(json.get(AstMappers.RANGE_KEY));
        } else {
            range = PositionRange.UNKNOWN;
        }
        return parser.parse(this, range, json);
    }

    <T extends Ast> List<T> parseArray(JsonNode json) {
        ArrayList<T> result = new ArrayList<>(json.size());
        json.elements().forEachRemaining(elem -> result.add(parse(elem)));
        return result;
    }

    public <T extends Ast> ObjectNode render(T ast) {
        ObjectNode node = mapper.createObjectNode();
        AstMapper<T> astMapper = AstMappers.mapperByClass(ast.getClass());
        node.put(AstMappers.TYPE_KEY, astMapper.getType());
        if (ast.getRange() != PositionRange.UNKNOWN && includeRanges) {
            node.putArray(AstMappers.RANGE_KEY).addAll(AstMappers.encodeRange(ast.getRange()));
        }
        astMapper.render(this, ast, node);
        return node;
    }

    public <T extends Ast> String renderToString(T ast) {
        AstMapper<T> astMapper = AstMappers.mapperByClass(ast.getClass());
        return astMapper.renderToString(this, ast);
    }

    public <T extends Ast> String renderToStringSubexpression(T ast) {
        AstMapper<T> astMapper = AstMappers.mapperByClass(ast.getClass());
        String rendered = astMapper.renderToString(this, ast);
        if (astMapper.needsParenthesesWhenUsedAsSubexpression()) {
            return '(' + rendered + ')';
        } else {
            return rendered;
        }
    }
}
