package ru.yandex.solomon.expression.compile;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.exceptions.CompilerException;
import ru.yandex.solomon.expression.expr.EvalContextImpl;
import ru.yandex.solomon.expression.expr.SelExpr;
import ru.yandex.solomon.expression.expr.SelExprVisitor;
import ru.yandex.solomon.expression.type.SelTypes;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class SelAssignment implements SelStatement {
    private final PositionRange range;
    private final String ident;
    private final SelExpr expr;

    SelAssignment(PositionRange range, String ident, SelExpr expr) {
        this.range = range;
        this.ident = ident;
        this.expr = expr;
    }

    @Override
    public PositionRange getRange() {
        return range;
    }

    public String getIdent() {
        return ident;
    }

    public SelExpr getExpr() {
        return expr;
    }

    @Override
    public String format() {
        return "let " + ident + " = " + expr.format() + ";";
    }

    @Override
    public SelStatement visit(SelExprVisitor visitor) {
        return new SelAssignment(range, ident, expr.visit(visitor));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        SelAssignment that = (SelAssignment) o;

        if (!this.getIdent().equals(that.getIdent())) return false;

        if (!this.getExpr().equals(that.getExpr())) return false;

        return true;
    }

    @Override
    public int hashCode() {
        return 31 * ident.hashCode() + expr.hashCode();
    }

    @Override
    public String toString() {
        return format();
    }

    @Override
    public void changeCompileContext(CompileContext compiler) {
        if (expr.type() == SelTypes.SIDE_EFFECT) {
            throw new CompilerException(expr.getRange(),
                "Result of " + expr + " cannot be assigned to a variable (attempted " + ident + ")");
        }
        compiler.addVar(ident, expr.type());
    }

    @Override
    public boolean changeEvalContext(EvalContextImpl context) {
        context.addVar(ident, expr.eval(context));
        return false;
    }
}
