package ru.yandex.solomon.expression.exceptions;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.jsonschema.JsonSchema;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.CaseFormat;

import ru.yandex.solomon.expression.Position;
import ru.yandex.solomon.expression.PositionRange;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public abstract class SelException extends RuntimeException {
    private final PositionRange range;
    private final String stage;
    private final String message;

    static final ObjectMapper mapper = new ObjectMapper();
    static final TypeReference<Map<String, Object>> DETAILS_TYPE = new TypeReference<>() {};

    static {
        mapper.disable(
                MapperFeature.AUTO_DETECT_CREATORS,
                MapperFeature.AUTO_DETECT_FIELDS,
                MapperFeature.AUTO_DETECT_GETTERS,
                MapperFeature.AUTO_DETECT_IS_GETTERS
        );
    }

    public static ObjectNode serializeRange(PositionRange range) {
        ObjectNode ret = mapper.createObjectNode();
        Position begin = range.getBegin();
        Position end = range.getEnd();

        ret.put("begin_offset", begin.getOffset());
        ret.put("begin_line", begin.getLine());
        ret.put("begin_column", begin.getColumn());
        ret.put("end_offset", end.getOffset());
        ret.put("end_line", end.getLine());
        ret.put("end_column", end.getColumn());

        return ret;
    }

    private SelException(PositionRange range, String stage, String message, Throwable cause) {
        super(stage + " error at " + range + ": " + message, cause);
        this.range = range;
        this.stage = stage;
        this.message = message;
    }

    protected SelException(PositionRange range, String stage, String message) {
        super(stage + " error at " + range + ": " + message);
        this.range = range;
        this.stage = stage;
        this.message = message;
    }

    protected SelException(PositionRange range, String stage, Throwable cause) {
        this(range, stage, cause.getClass().getSimpleName() + ": " + cause.getMessage(), cause);
    }

    public PositionRange getRange() {
        return range;
    }

    public String getStage() {
        return stage;
    }

    public String getErrorMessage() {
        return message;
    }

    private static String getClassNameAsUpperCase(Class<? extends SelException> clazz) {
        return CaseFormat.UPPER_CAMEL.to(CaseFormat.UPPER_UNDERSCORE, clazz.getSimpleName());
    }

    public String getType() {
        return getClassNameAsUpperCase(getClass());
    }

    public Map<String, Object> getDetails() {
        HashMap<String, Object> ret = new HashMap<>(mapper.convertValue(this, DETAILS_TYPE));

        String classCaps = stage.toUpperCase() + "_ERROR";
        ret.put("class", classCaps);
        if (!PositionRange.UNKNOWN.equals(range)) {
            ret.put("range", serializeRange(range));
        }

        return ret;
    }

    public static class ExceptionSchema {
        private final String type;
        private final ObjectNode schema;

        public ExceptionSchema(String type, JsonSchema schema) {
            this.type = type;
            this.schema = expandSchema(validateSchema(schema.getSchemaNode()));
        }

        private static ObjectNode expandSchema(ObjectNode schema) {
            ObjectNode properties = (ObjectNode) schema.findValue("properties");
            ObjectNode typeInteger = mapper.createObjectNode().put("type", "integer");
            ObjectNode rangeProperties = mapper.createObjectNode();
            rangeProperties.set("begin_offset", typeInteger);
            rangeProperties.set("begin_line", typeInteger);
            rangeProperties.set("begin_column", typeInteger);
            rangeProperties.set("end_offset", typeInteger);
            rangeProperties.set("end_line", typeInteger);
            rangeProperties.set("end_column", typeInteger);

            properties.with("range")
                    .put("type", "object")
                    .put("required", false)
                    .set("properties", rangeProperties);

            properties.with("class")
                    .put("type", "string");

            return schema;
        }

        private static final Set<String> FORBIDDEN_FIELDS = Set.of(
                "class", // Exception category in ALL_CAPS
                "range", // Position in program
                ""
        );

        private ObjectNode validateSchema(ObjectNode schemaNode) {
            JsonNode properties = schemaNode.findValue("properties");
            for (var field : FORBIDDEN_FIELDS) {
                if (properties.has(field)) {
                    throw new IllegalArgumentException("Field `" + field + "' is a reserved name, found in " + type);
                }
            }
            return schemaNode;
        }

        public String getType() {
            return type;
        }

        public ObjectNode getSchema() {
            return schema;
        }

        @Override
        public String toString() {
            return "Schema{" +
                    "type='" + type + '\'' +
                    ", schema=" + schema +
                    '}';
        }
    }

    public static ExceptionSchema getExceptionSchema(Class<? extends SelException> clazz) {
        String type = getClassNameAsUpperCase(clazz);
        // TODO: migrate to jackson-module-jsonSchema module (requires matched jackson version)
        try {
            return new ExceptionSchema(type, mapper.generateJsonSchema(clazz));
        } catch (JsonMappingException e) {
            throw new RuntimeException(e);
        }
    }
}
