package ru.yandex.solomon.expression.exceptions;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class TooManyLinesInAggregation extends EvaluationException {
    @JsonProperty("aggregation_function")
    public final String aggregationFunction;

    @JsonProperty("series_labels")
    public final List<String> seriesLabels;

    public TooManyLinesInAggregation(PositionRange range, String funcName, List<NamedGraphData> namedGraphData) {
        super(range, makeMessage(funcName, namedGraphData));
        this.aggregationFunction = funcName;
        this.seriesLabels = namedGraphData.stream()
                .map(NamedGraphData::getLabels)
                .map(Labels::toString)
                .collect(Collectors.toList());
    }

    private static String makeMessage(String funcName, List<NamedGraphData> namedGraphData) {
        StringBuilder errorMessage = new StringBuilder();
        errorMessage.append("Not able to apply function ")
                .append(funcName)
                .append(" on vector with size ")
                .append(namedGraphData.size());

        List<String> allLabels = namedGraphData.stream()
                .map(NamedGraphData::getLabels)
                .filter(labels -> !labels.isEmpty())
                .map(Labels::toString)
                .collect(Collectors.toList());

        int headLen = 3;
        int tailLen = 3;

        if (allLabels.size() > headLen + tailLen) {
            List<String> collapsed = allLabels.stream()
                    .limit(headLen)
                    .collect(Collectors.toCollection(ArrayList::new));
            collapsed.add("...");
            allLabels = allLabels.stream()
                    .skip(allLabels.size() - tailLen)
                    .collect(Collectors.toCollection(() -> collapsed));
        }

        String allLabelsString = String.join(",\n", allLabels);

        if (!allLabels.isEmpty()) {
            errorMessage.append(" [\n").append(allLabelsString).append("]");
        }

        return errorMessage.toString();
    }
}
