package ru.yandex.solomon.expression.expr;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.analytics.GraphDataLoadRequest;
import ru.yandex.solomon.expression.analytics.GraphDataLoader;
import ru.yandex.solomon.expression.compile.CompileContext;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueSideEffect;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.util.time.Interval;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class EvalContextImpl implements EvalContext {
    private final Interval interval;
    private final HashMap<String, SelValue> params;
    private final GraphDataLoader loader;
    private final SideEffectProcessor sideEffectProcessor;
    private final SelVersion version;

    public EvalContextImpl(
            Interval interval,
            Map<String, SelValue> params,
            GraphDataLoader loader,
            SideEffectProcessor processor,
            SelVersion version)
    {
        this.interval = interval;
        this.params = new HashMap<>(params);
        this.loader = loader;
        this.sideEffectProcessor = processor;
        this.version = version;
    }

    public EvalContextImpl(Map<String, SelValue> params, SelVersion version) {
        this(Interval.millis(0, 1000), params, GraphDataLoader.empty(), SideEffectProcessor.throwing(), version);
    }

    public SelValue getOrThrow(String name) {
        return Objects.requireNonNull(params.get(name));
    }

    public SelValue getOrNull(String name) {
        return params.get(name);
    }

    public CompileContext compileContext() {
        Map<String, SelType> types = params.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().type()));
        return new CompileContext(types);
    }

    @Override
    public Interval getInterval() {
        return interval;
    }

    public void addVar(String key, SelValue value) {
        params.put(key, value);
    }

    public Map<String, SelValue> getVars() {
        return params;
    }

    public SelValue loadGraphData(PositionRange range, GraphDataLoadRequest request) {
        NamedGraphData[] result = loader.loadGraphData(request);
        if (request.getType() == SelTypes.GRAPH_DATA) {
            if (result.length == 0) {
                throw new EvaluationException(range,
                    "no lines found for request: {" + Selectors.format(request.getSelectors()) + "}");
            } else if (result.length == 1) {
                return new SelValueGraphData(result[0]);
            } else {
                String labels = Stream.of(result)
                        .map(gd -> gd.getLabels().toString())
                        .collect(Collectors.joining("; "));

                throw new EvaluationException(range,
                    "selector {" + Selectors.format(request.getSelectors()) + "} was resolved into multiple lines " +
                    "while only one line was allowed: " + labels);
            }
        } else {
            SelValue[] vectorData = Arrays.stream(result).map(SelValueGraphData::new).toArray(SelValue[]::new);
            return new SelValueVector(SelTypes.GRAPH_DATA, vectorData);
        }
    }

    public void process(SelValueSideEffect sideEffect) {
        this.sideEffectProcessor.process(sideEffect);
    }

    @Override
    public SelVersion getVersion() {
        return version;
    }

    @Override
    public long getGridMillis() {
        return loader.getSeriesGridMillis();
    }
}
