package ru.yandex.solomon.expression.expr;

import java.util.HashMap;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.solomon.expression.ast.AstIdent;
import ru.yandex.solomon.expression.compile.SelStatement;
import ru.yandex.solomon.expression.exceptions.PreparingException;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class IntrinsicsExternalizer {
    private final Interval interval;

    @Nullable
    private final ShardKey alertingStatuses;
    @Nullable
    private final String alertId, projectId, parentId;

    private final HashMap<SelExprParam, Selectors> intrinsicSelectors;

    private IntrinsicsExternalizer(Builder builder) {
        interval = builder.interval;
        alertingStatuses = builder.alertingStatuses;
        alertId = builder.alertId;
        projectId = builder.projectId;
        parentId = builder.parentId;
        intrinsicSelectors = new HashMap<>();
    }

    public List<SelStatement> fillIntrinsics(SelVersion version, List<SelStatement> block) {
        MutableInt callCounter = new MutableInt(0);
        return SelExprFnExternalizeVisitor.replaceFnCall(
            IntrinsicsExternalizer::isIntrinsicFn,
            call -> externalizeIntrinsic(version, call, callCounter),
            block
        );
    }

    private static boolean isIntrinsicFn(SelExprFuncCall fnCall) {
        return true;
    }

    private SelExpr externalizeIntrinsic(SelVersion version, SelExprFuncCall fnCall, MutableInt callCounter) {
        var fn = fnCall.getFunc();
        var ex = fn.getExternalizer();
        if (ex == null) {
            return fnCall;
        }
        ArgsList values = new ArgsList(fnCall.getRange(), fnCall.getArgs().size());
        for (var arg : fnCall.getArgs()) {
            if (arg instanceof SelExprValue selExprValue) {
                values.add(selExprValue::getValue, selExprValue.getRange());
            } else {
                throw new PreparingException(arg.getRange(), "Only compile-time constant values are allowed");
            }
        }
        Selectors externalizedSelectors = ex.apply(values, this);
        if (externalizedSelectors == null) {
            return fnCall;
        }

        String varName = "intrinsicSel$" + fnCall.getFunc().getName() + '$' + callCounter.getAndAdd(1);
        AstIdent astIdent = new AstIdent(fnCall.getRange(), varName);
        SelExprParam param = new SelExprParam(astIdent, fnCall.type());
        intrinsicSelectors.put(param, externalizedSelectors);
        return fn.getExternalizerPostProcess().apply(version, values, param);
    }

    public HashMap<SelExprParam, Selectors> getIntrinsicSelectors() {
        return intrinsicSelectors;
    }

    public Interval getInterval() {
        return interval;
    }

    @Nullable
    public ShardKey getAlertingStatuses() {
        return alertingStatuses;
    }

    @Nullable
    public String getAlertId() {
        return alertId;
    }

    @Nullable
    public String getProjectId() {
        return projectId;
    }

    @Nullable
    public String getParentId() {
        return parentId;
    }

    public static Builder newBuilder(Interval interval) {
        return new Builder(interval);
    }

    @ParametersAreNonnullByDefault
    public static class Builder {
        private Interval interval;
        private ShardKey alertingStatuses;
        private String alertId, projectId, parentId;

        public Builder(Interval interval) {
            this.interval = interval;
        }

        public Builder setAlertingStatuses(ShardKey alertingStatuses) {
            this.alertingStatuses = alertingStatuses;
            return this;
        }

        public Builder setAlertKey(String alertId, String projectId, String parentId) {
            this.alertId = alertId;
            this.projectId = projectId;
            this.parentId = parentId;
            return this;
        }

        public IntrinsicsExternalizer build() {
            return new IntrinsicsExternalizer(this);
        }
    }
}
