package ru.yandex.solomon.expression.expr;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;

import ru.yandex.solomon.expression.compile.SelAssignment;
import ru.yandex.solomon.expression.compile.SelStatement;
import ru.yandex.solomon.expression.expr.func.GraphDataSelFnIntegrate;
import ru.yandex.solomon.expression.expr.func.SelFnDerivative;
import ru.yandex.solomon.expression.expr.func.SelFnDiff;
import ru.yandex.solomon.expression.expr.func.SelFnNonNegativeDerivative;
import ru.yandex.solomon.expression.expr.func.analytical.SelFnAsap;
import ru.yandex.solomon.expression.expr.func.analytical.SelFnCrop;
import ru.yandex.solomon.expression.expr.func.analytical.SelFnMovingAvg;
import ru.yandex.solomon.expression.expr.func.analytical.SelFnMovingPercentile;
import ru.yandex.solomon.expression.expr.func.analytical.SelFnMovingSum;
import ru.yandex.solomon.expression.expr.func.analytical.rank.SelFnRank;

/**
 * It's necessary to compute program type and optimize load requests for every program type.
 * It's designed for __oldMode support in Data API specially (see SOLOMON-5053), but you can use it in other places.
 *
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ProgramTypeVisitor {
    private static final Set<String> availableTransformationFunctions;

    static {
        HashSet<String> names = new HashSet<>();

        names.add(SelFnNonNegativeDerivative.NAME);
        names.add(SelFnDerivative.NAME);
        names.add(SelFnDiff.NAME);
        names.add(SelFnMovingAvg.NAME);
        names.add(SelFnMovingSum.NAME);
        names.add(SelFnMovingPercentile.NAME);
        names.add(SelFnAsap.NAME);
        names.add(SelFnCrop.NAME);
        names.addAll(GraphDataSelFnIntegrate.NAMES);

        availableTransformationFunctions = ImmutableSet.copyOf(names);
    }

    private static ProgramType checkExpr(SelExpr expr) {
        if (expr instanceof SelExprSelectors) {
            return ProgramType.SELECTORS;
        }

        if (expr instanceof SelExprFuncCall funcCallExpr) {
            String funcName = funcCallExpr.getFunc().getName();
            if (SelFnRank.matches(funcName)) {
                if (isComplexArgs(funcCallExpr)) {
                    return ProgramType.COMPLEX;
                }
                return ProgramType.TOP;
            }
            if (availableTransformationFunctions.contains(funcName)) {
                if (isComplexArgs(funcCallExpr)) {
                    return ProgramType.COMPLEX;
                }
                return ProgramType.SIMPLE;
            } else {
                // Not available function
                return ProgramType.COMPLEX;
            }
        }

        return ProgramType.COMPLEX;
    }

    private static boolean isComplexArgs(SelExprFuncCall funcCallExpr) {
        for (SelExpr argExpr : funcCallExpr.getArgs()) {
            if (!(argExpr instanceof SelExprValue || argExpr instanceof SelExprSelectors)) {
                return true;
            }
        }
        return false;
    }

    public static ProgramType computeProgramType(List<SelStatement> statements) {
        if (statements.size() != 1) {
            return ProgramType.COMPLEX;
        }

        SelStatement selStatement = statements.get(0);

        if (selStatement instanceof SelAssignment selAssignment) {
            SelExpr expr = selAssignment.getExpr();
            return checkExpr(expr);
        }

        return ProgramType.COMPLEX;
    }
}
