package ru.yandex.solomon.expression.expr;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.ExpressionMetrics.FnMetrics;
import ru.yandex.solomon.expression.ast.AstCall;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.exceptions.PreparingException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelExprFuncCall extends SelExpr {
    private final SelFunc func;
    private final List<SelExpr> args;
    private final FnMetrics metrics;

    public SelExprFuncCall(AstCall call, SelFunc func, List<SelExpr> args, FnMetrics metrics) {
        super(call);
        this.func = func;
        this.args = args;
        this.metrics = metrics;
    }

    @Override
    public AstCall getSourceAst() {
        return (AstCall) super.getSourceAst();
    }

    @Nonnull
    @Override
    public SelType type() {
        return func.getReturnType();
    }

    public SelFunc getFunc() {
        return func;
    }

    public List<SelExpr> getArgs() {
        return args;
    }

    @Nonnull
    @Override
    public SelValue evalInternal(EvalContextImpl context) {
        metrics.calls.inc();
        ArgsList argsList = new ArgsList(getRange(), args.size());
        for (SelExpr expr : args) {
            argsList.add(() -> expr.eval(context), expr.getRange());
        }
        SelValue result = func.getHandler().apply(context, argsList);
        if (!result.type().equals(type())) {
            throw new EvaluationException(getRange(),
                "Evaluation result type '" + result.type() + "' mismatches declared expression type '" + type());
        }
        return result;
    }

    @Override
    protected SelExprFuncCall mapParams(ParamsMapper f) {
        List<SelExpr> update = new ArrayList<>(args.size());
        for (SelExpr expr : args) {
            var updated = f.apply(expr);
            if (!Objects.equals(expr.type(), updated.type())) {
                throw new PreparingException(expr.getRange(),
                    "Unable to change argument type, " + expr.type() + " -> " + updated.type());
            }
            update.add(updated);
        }
        return new SelExprFuncCall(getSourceAst(), func, update, metrics);
    }

    @Override
    public SelExpr visit(SelExprVisitor visitor) {
        return visitor.visitFn(this);
    }

    @Override
    public String format() {
        return func.getName() + "(" + args.stream().map(SelExpr::format).collect(Collectors.joining(", ")) + ")";
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SelExprFuncCall that = (SelExprFuncCall) o;
        return func.equals(that.func) &&
            args.equals(that.args);
    }

    @Override
    public int hashCode() {
        return Objects.hash(func, args);
    }
}
