package ru.yandex.solomon.expression.expr;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.ast.AstObject;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueObject;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class SelExprObject extends SelExpr {
    private final Map<String, SelExpr> object;

    public SelExprObject(AstObject ast, Map<String, SelExpr> object) {
        super(ast);
        this.object = object;
    }

    @Override
    public AstObject getSourceAst() {
        return (AstObject) super.getSourceAst();
    }

    public Map<String, SelExpr> getObject() {
        return object;
    }

    @Override
    public SelExpr mapParams(ParamsMapper f) {
        Map<String, SelExpr> update = new HashMap<>(object.size());
        for (Map.Entry<String, SelExpr> entry : object.entrySet()) {
            update.put(entry.getKey(), f.apply(entry.getValue()));
        }
        return new SelExprObject(getSourceAst(), update);
    }

    @Nonnull
    @Override
    public SelType type() {
        return SelTypes.OBJECT;
    }

    @Nonnull
    @Override
    public SelValue evalInternal(EvalContextImpl context) {
        Map<String, SelValue> result = new HashMap<>(object.size());
        for (Map.Entry<String, SelExpr> entry : object.entrySet()) {
            result.put(entry.getKey(), entry.getValue().eval(context));
        }
        return new SelValueObject(result);
    }

    @Override
    public SelExpr visit(SelExprVisitor visitor) {
        return visitor.visitObject(this);
    }

    @Override
    public String format() {
        String content = this.object.entrySet().stream()
            .map(entry -> entry.getKey() + ": " + entry.getValue())
            .collect(Collectors.joining(", "));

        return '{' + content + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SelExprObject that = (SelExprObject) o;
        return object.equals(that.object);
    }

    @Override
    public int hashCode() {
        return Objects.hash(object);
    }
}
