package ru.yandex.solomon.expression.expr;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.ExpressionMetrics.OpMetrics;
import ru.yandex.solomon.expression.ast.Ast;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.exceptions.PreparingException;
import ru.yandex.solomon.expression.expr.func.SelOp;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class SelExprOpCall extends SelExpr {
    private final SelOp op;
    private final List<SelExpr> params;
    private final OpMetrics metrics;

    public SelExprOpCall(Ast astOp, SelOp op, List<SelExpr> params, OpMetrics metrics) {
        super(astOp);
        this.op = op;
        this.params = params;
        this.metrics = metrics;
    }

    public List<SelExpr> getParams() {
        return params;
    }

    @Override
    public SelExprOpCall mapParams(ParamsMapper f) {
        List<SelExpr> update = new ArrayList<>(params.size());
        for (SelExpr expr : params) {
            var updated = f.apply(expr);
            if (!Objects.equals(expr.type(), updated.type())) {
                throw new PreparingException(getRange(), "Unable to change argument types, " + expr.type() + " -> " + updated.type());
            }
            update.add(updated);
        }
        return new SelExprOpCall(getSourceAst(), op, update, metrics);
    }

    @Nonnull
    @Override
    public SelType type() {
        return op.getReturnType();
    }

    @Nonnull
    @Override
    public SelValue evalInternal(EvalContextImpl context) {
        metrics.calls.inc();
        ArgsList args = new ArgsList(getRange(), params.size());
        for (SelExpr expr : params) {
            args.add(() -> expr.eval(context), expr.getRange());
        }
        SelValue result = op.getHandler().apply(context, args);
        if (!result.type().equals(type())) {
            throw new EvaluationException(getRange(),
                "Evaluation result type '" + result.type() + "' mismatches declared expression type '" + type());
        }
        return result;
    }

    @Override
    public SelExpr visit(SelExprVisitor visitor) {
        return visitor.visitOp(this);
    }

    @Override
    public String format() {
        return SelOp.format(op.getOperator(), params);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SelExprOpCall that = (SelExprOpCall) o;
        return op.equals(that.op) &&
            params.equals(that.params);
    }

    @Override
    public int hashCode() {
        return Objects.hash(op, params);
    }
}
