package ru.yandex.solomon.expression.expr;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.analytics.GraphDataLoadRequest;
import ru.yandex.solomon.expression.ast.AstValueString;
import ru.yandex.solomon.expression.compile.SelStatement;
import ru.yandex.solomon.expression.expr.func.analytical.rank.SelFnRank;
import ru.yandex.solomon.expression.value.SelValueString;
import ru.yandex.solomon.math.protobuf.Aggregation;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public final class SelExprRankExternalizer {
    private static final Map<String, Aggregation> AGGREGATION_BY_NAME = Arrays.stream(Aggregation.values())
            .filter(aggr -> aggr != Aggregation.DEFAULT_AGGREGATION && aggr != Aggregation.UNRECOGNIZED)
            .collect(Collectors.toMap(
                    aggr -> aggr.name().toLowerCase(),
                    aggr -> aggr
            ));


    private SelExprRankExternalizer() {
    }

    public static List<SelStatement> fillRank(
            Map<SelExprParam, GraphDataLoadRequest.Builder> extVarToRequest,
            List<SelStatement> block)
    {
        return SelExprFnExternalizeVisitor.replaceFnCall(
            SelExprRankExternalizer::isRankFn,
            fnCall -> externalizeRank(extVarToRequest, fnCall),
            block
        );
    }

    private static boolean isRankFn(SelExprFuncCall fnCall) {
        String name = fnCall.getFunc().getName();
        return SelFnRank.matches(name);
    }

    private static SelExpr externalizeRank(
            Map<SelExprParam, GraphDataLoadRequest.Builder> extVarToRequest,
            SelExprFuncCall fnCall)
    {
        String funcName = fnCall.getFunc().getName();
        boolean asc = funcName.startsWith("bottom");
        var args = fnCall.getArgs();

        SelExpr dataParam = args.get(args.size() - 1);
        if (!(dataParam instanceof SelExprParam)) {
            return fnCall;
        }

        GraphDataLoadRequest.Builder builder = extVarToRequest.get(dataParam);
        if (builder == null) {
            return fnCall;
        }

        SelExpr limitParam = args.get(0);
        if (!(limitParam instanceof SelExprValue)) {
            return fnCall;
        }

        int limit = (int) ((SelExprValue) limitParam).getValue().castToScalar().getValue();

        SelExpr aggrParam;
        int underscore = funcName.indexOf('_');
        if (underscore == -1) {
            aggrParam = fnCall.getArgs().get(1);
        } else {
            String suffix = funcName.substring(underscore + 1);
            AstValueString aggr = new AstValueString(fnCall.getRange(), suffix);
            aggrParam = new SelExprValue(aggr, new SelValueString(suffix));
        }
        if (!(aggrParam instanceof SelExprValue)) {
            return fnCall;
        }
        String aggrParamValue = ((SelExprValue) aggrParam).getValue().castToString().getValue();

        @Nullable Aggregation aggrFnOpt = AGGREGATION_BY_NAME.get(aggrParamValue);

        if (aggrFnOpt == null) {
            return fnCall;
        }

        builder.setRankFilter(asc, limit, aggrFnOpt);

        // Don't eliminate call since cross-shard top is splitted into several readMany requests
        // and extra filtration is nessesary. For single shard the function will short-circuit
        return fnCall;
    }
}
