package ru.yandex.solomon.expression.expr;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.AstIdent;
import ru.yandex.solomon.expression.exceptions.PreparingException;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsBuilder;
import ru.yandex.solomon.labels.selector.LabelSelectorSet;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelExprSelectorsReplaceVisitor extends SelExprRecurseVisitor {
    private final Map<SelExprParam, Selectors> selectorByParam = new HashMap<>();
    private final boolean useNewFormat;

    public SelExprSelectorsReplaceVisitor(boolean useNewFormat) {
        this.useNewFormat = useNewFormat;
    }

    @Override
    public SelExpr visitFn(SelExprFuncCall fnCall) {
        var fn = fnCall.getFunc();
        if (fn.getName().equals("load") || fn.getName().equals("load1")) {
            String selector = ((SelExprValue) fnCall.getArgs().get(0))
                .getValue()
                .castToString()
                .getValue();

            String varName = "prefetch$" + fn.getName() + "_" + selectorByParam.size();
            AstIdent varIdent = new AstIdent(fnCall.getRange(), varName);
            SelExprParam replace = new SelExprParam(varIdent, fnCall.type());
            selectorByParam.put(replace, LabelSelectorSet.parseEscaped(selector).toNewSelectors());
            return replace;
        }
        return super.visitFn(fnCall);
    }

    @Override
    public SelExpr visitSelectors(SelExprSelectors selectors) {
        String varName = "prefetch$selector_" + selectorByParam.size();
        AstIdent varIdent = new AstIdent(selectors.getRange(), varName);
        SelExprParam replace = new SelExprParam(varIdent, selectors.type());

        final SelectorsBuilder result;

        if (useNewFormat) {
            result = Selectors.builder(selectors.getNameSelector(), selectors.getSelectors().size());
        } else {
            result = Selectors.builder(selectors.getSelectors().size() + 1);

            if (!selectors.getNameSelector().isEmpty()) {
                result.add(Selector.exact(LabelKeys.SENSOR, selectors.getNameSelector()));
            }
        }

        for (SelExprSelector selector : selectors.getSelectors()) {
            String key = getConstantString(selector.getKey());
            String value = getConstantString(selector.getValue());
            if (key == null || value == null) {
                throw new PreparingException(
                    PositionRange.convexHull(selector.getKey().getRange(), selector.getValue().getRange()),
                    "Selector should contain only constant values");
            }
            result.add(selector.getType().create(key, value));
        }
        selectorByParam.put(replace, result.build());
        return replace;
    }

    @Nullable
    private String getConstantString(SelExpr expr) {
        if (expr instanceof SelExprValue) {
            SelValue value = ((SelExprValue) expr).getValue();
            return value.convertToString();
        }

        return null;
    }

    public Map<SelExprParam, Selectors> getSelectorByParam() {
        return selectorByParam;
    }
}
