package ru.yandex.solomon.expression.expr;

import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.AstTernaryOp;
import ru.yandex.solomon.expression.exceptions.CompilerException;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.exceptions.PreparingException;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelExprTern extends SelExpr {
    private final SelExpr condition;
    private final SelExpr ifTrue;
    private final SelExpr ifFalse;

    public SelExprTern(AstTernaryOp sourceAst, SelExpr condition, SelExpr ifTrue, SelExpr ifFalse) {
        super(sourceAst);
        this.condition = condition;
        this.ifTrue = ifTrue;
        this.ifFalse = ifFalse;
        if (condition.type() != SelTypes.BOOLEAN) {
            throw new CompilerException(condition.getRange(), "Condition should be Boolean but was: " + condition.type());
        }
        if (!ifTrue.type().equals(ifFalse.type())) {
            throw new CompilerException(
                PositionRange.convexHull(ifFalse.getRange(), ifTrue.getRange()),
                "Ternary operator operands have different types: " + ifTrue.type() + " and " + ifFalse.type());
        }
    }

    @Override
    public AstTernaryOp getSourceAst() {
        return (AstTernaryOp) super.getSourceAst();
    }

    @Nonnull
    @Override
    public SelType type() {
        return ifTrue.type();
    }

    @Nonnull
    @Override
    public SelValue evalInternal(EvalContextImpl context) {
        SelValue conditionCheck = condition.evalInternal(context);
        if (conditionCheck.castToBoolean().getValue()) {
            return ensureValidReturnType(ifTrue.eval(context));
        }
        return ensureValidReturnType(ifFalse.eval(context));
    }

    private SelValue ensureValidReturnType(SelValue result) {
        if (!result.type().equals(type())) {
            throw new EvaluationException(getRange(),
                "Evaluation result type '" + result.type() + "' mismatches declared expression type '" + type());
        }
        return result;
    }


    @Override
    protected SelExprTern mapParams(ParamsMapper f) {
        try {
            return new SelExprTern(getSourceAst(), f.apply(condition), f.apply(ifTrue), f.apply(ifFalse));
        } catch (CompilerException e) {
            throw new PreparingException(e.getRange(), "Trying to change TernOp argument types: " + e.getErrorMessage());
        }
    }

    public SelExpr getCondition() {
        return condition;
    }

    public SelExpr getIfTrue() {
        return ifTrue;
    }

    public SelExpr getIfFalse() {
        return ifFalse;
    }

    @Override
    public SelExpr visit(SelExprVisitor visitor) {
        return visitor.visitCondition(this);
    }

    @Override
    public String format() {
        return condition + " ? " + ifTrue + " : " + ifFalse;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SelExprTern that = (SelExprTern) o;
        return condition.equals(that.condition) &&
            ifTrue.equals(that.ifTrue) &&
            ifFalse.equals(that.ifFalse);
    }

    @Override
    public int hashCode() {
        return Objects.hash(condition, ifTrue, ifFalse);
    }
}
