package ru.yandex.solomon.expression.expr.func;

import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.exceptions.TooManyLinesInAggregation;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnAvg;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnCount;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnIntegrate;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnInterquartileRange;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnLast;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnMax;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnMedian;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnMin;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnRandom;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnStd;
import ru.yandex.solomon.expression.expr.func.aggr.AggrSelFnSum;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueDouble;
import ru.yandex.solomon.math.doubles.DoubleTimeSeriesFactory;
import ru.yandex.solomon.model.timeseries.view.DoubleTimeSeriesView;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class AggrSelFn implements SelFuncProvider {
    protected abstract String name();

    public abstract double evalArrayView(DoubleArrayView param);

    public final double evalArray(double[] array) {
        return evalArrayView(new DoubleArrayView(array));
    }

    public double evalGraphData(NamedGraphData source) {
        return evalTimeSeries(DoubleTimeSeriesFactory.fromGraphData(
                source.getDataType(), source.getAggrGraphDataArrayList()));
    }

    public double evalTimeSeries(DoubleTimeSeriesView timeSeriesView) {
        return evalArrayView(timeSeriesView.getValueView());
    }

    private SelValue evalOnGraphData(PositionRange range, SelValue[] values) {
        if (values.length != 1) {
            throw new TooManyLinesInAggregation(range, name(), Stream.of(values)
                    .map(v -> v.castToGraphData().getNamedGraphData())
                    .collect(Collectors.toList())
            );
        }

        return new SelValueDouble(evalGraphData(values[0].castToGraphData().getNamedGraphData()));
    }

    public boolean isPure() {
        return true;
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name(name())
            .help("Aggregate values")
            .category(SelFuncCategory.AGGREGATION)
            .args(SelTypes.DOUBLE_VECTOR)
            .returnType(SelTypes.DOUBLE)
            .pure(isPure())
            .handler(args -> {
                var source = args.get(0).castToVector().doubleArray();
                return new SelValueDouble(evalArray(source));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name(name())
            .help("Aggregate values")
            .category(SelFuncCategory.AGGREGATION)
            .args(SelTypes.GRAPH_DATA)
            .returnType(SelTypes.DOUBLE)
            .pure(isPure())
            .handler(args -> {
                var source = args.get(0).castToGraphData().getNamedGraphData();
                return new SelValueDouble(evalGraphData(source));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name(name())
            .help("Aggregate values")
            .category(SelFuncCategory.AGGREGATION)
            .args(SelTypes.GRAPH_DATA_VECTOR)
            .returnType(SelTypes.DOUBLE)
            .pure(isPure())
            .handler(args -> {
                var source = args.get(0).castToVector().valueArray();
                return evalOnGraphData(args.getRange(0), source);
            })
            .build());
    }

    public enum Type {
        MAX(new AggrSelFnMax()),
        MIN(new AggrSelFnMin()),
        AVG(new AggrSelFnAvg()),
        SUM(new AggrSelFnSum()),
        LAST(new AggrSelFnLast()),
        STD(new AggrSelFnStd()),
        COUNT(new AggrSelFnCount()),
        INTEGRATE(new AggrSelFnIntegrate()),
        RANDOM(new AggrSelFnRandom()),
        MEDIAN(new AggrSelFnMedian()),
        IQR(new AggrSelFnInterquartileRange())
        ;

        private final AggrSelFn func;

        Type(AggrSelFn func) {
            this.func = func;
        }

        public AggrSelFn getFunc() {
            return func;
        }

        public static Optional<Type> byName(String name) {
            for (Type type : values()) {
                if (type.func.name().equals(name)) {
                    return Optional.of(type);
                }
            }
            return Optional.empty();
        }
    }
}
