package ru.yandex.solomon.expression.expr.func;

import java.util.Optional;
import java.util.function.DoubleUnaryOperator;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValues;
import ru.yandex.solomon.math.doubles.DoubleMath;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class DoubleSelFn implements SelFuncProvider {
    protected abstract String name();

    protected abstract String help();

    protected SelFuncCategory category() {
        return SelFuncCategory.TRANSFORMATION;
    }

    public abstract double eval(double p);

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name(name())
            .help(help())
            .category(category())
            .args(arg("source").type(SelTypes.DOUBLE))
            .returnType(SelTypes.DOUBLE)
            .handler(args -> SelValues.mapAsDouble(args.get(0).castToScalar(), this::eval))
            .build());

        registry.add(SelFunc.newBuilder()
            .name(name())
            .help(help())
            .category(category())
            .args(arg("source").type(SelTypes.DOUBLE_VECTOR))
            .returnType(SelTypes.DOUBLE_VECTOR)
            .handler(args -> SelValues.mapArrayAsDouble(args.get(0).castToVector(), this::eval))
            .build());

        registry.add(SelFunc.newBuilder()
            .name(name())
            .help(help())
            .category(category())
            .args(arg("source").type(SelTypes.GRAPH_DATA).help("applies on each point value"))
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> SelValues.mapValueAsDouble(args.get(0).castToGraphData(), this::eval))
            .build());

        registry.add(SelFunc.newBuilder()
            .name(name())
            .help(help())
            .category(category())
            .args(arg("source").type(SelTypes.GRAPH_DATA_VECTOR).help("applied on each point value"))
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> SelValues.mapValuesAsDouble(args.get(0).castToVector(), this::eval))
            .build());
    }

    public enum Type {
        SQR("square of the value", x -> x * x, SelFuncCategory.DEPRECATED),
        SQRT("square root of the value", Math::sqrt),
        ABS("absolute value", Math::abs),
        SIGN("signum of the value", Math::signum),
        HEAVISIDE("replace positive value with 1, negative and zero with 0", DoubleMath::heaviside),
        RAMP("replace negative value with 0", DoubleMath::ramp),
        EXP("computes exponential function", Math::exp),
        LOG("computes natural logarithm", Math::log),
        FLOOR("rounds value down", Math::floor),
        CEIL("rounds value up", Math::ceil),
        ROUND("rounds value to the nearest integer", x -> Math.floor(x + 0.5)),
        TRUNC("rounds value towards zero", DoubleMath::trunc),
        FRACT("value fractional part", DoubleMath::fract),
        ;

        private final DoubleSelFn func;

        Type(String help, DoubleUnaryOperator f) {
            final String name = Type.this.name().toLowerCase();
            this.func = new DoubleSelFn() {
                @Override
                protected String name() {
                    return name;
                }

                @Override
                protected String help() {
                    return help;
                }

                @Override
                public double eval(double p) {
                    return f.applyAsDouble(p);
                }
            };
        }

        Type(String help, DoubleUnaryOperator f, SelFuncCategory category) {
            final String name = Type.this.name().toLowerCase();
            this.func = new DoubleSelFn() {
                @Override
                protected String name() {
                    return name;
                }

                @Override
                protected String help() {
                    return help;
                }

                @Override
                public double eval(double p) {
                    return f.applyAsDouble(p);
                }

                @Override
                public SelFuncCategory category() {
                    return category;
                }
            };
        }

        public static Optional<Type> byName(String name) {
            for (Type type : values()) {
                if (type.func.name().equals(name)) {
                    return Optional.of(type);
                }
            }
            return Optional.empty();
        }

        public DoubleSelFn getFunc() {
            return func;
        }
    }
}
