package ru.yandex.solomon.expression.expr.func;

import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.EvalContext;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.GraphDataArrayList;

/**
 * Returns constant line from begin to end of time interval range.
 * Example: <code>constant_line(120)</code>
 *
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class SelFnConstantLine implements SelFuncProvider {

    public static final String NAME = "constant_line";

    @Override
    public void provide(SelFuncRegistry registry) {
        provide(registry, SelVersion.GROUP_LINES_RETURN_VECTOR_2::before,
                SelTypes.GRAPH_DATA, SelFnConstantLine::scalarLine);
        provide(registry, SelVersion.GROUP_LINES_RETURN_VECTOR_2::since,
                SelTypes.GRAPH_DATA_VECTOR, SelFnConstantLine::vectorLine);
    }

    private void provide(
            SelFuncRegistry registry,
            Predicate<SelVersion> supportedVersions,
            SelType returnType,
            SelFunctionWithContext handler)
    {
        registry.add(SelFunc.newBuilder()
            .name(NAME)
            .help("Constant line by time interval range")
            .category(SelFuncCategory.OTHER)
            .args(SelTypes.DOUBLE)
            .pure(false)
            .supportedVersions(supportedVersions)
            .returnType(returnType)
            .handler(handler)
            .build());

        registry.add(SelFunc.newBuilder()
                .name(NAME)
                .help("Constant line by time interval range with fixed grid")
                .category(SelFuncCategory.OTHER)
                .args(SelTypes.DOUBLE, SelTypes.DURATION)
                .pure(false)
                .supportedVersions(supportedVersions)
                .returnType(returnType)
                .handler(handler)
                .build());
    }

    private static SelValue scalarLine(EvalContext ctx, ArgsList args) {
        double constant = args.get(0).castToScalar().getValue();
        long begin = ctx.getInterval().getBeginMillis();
        long end = ctx.getInterval().getEndMillis();
        long gridMillis = args.size() == 1 ? ctx.getGridMillis() : args.get(1).castToDuration().getDuration().toMillis();
        if (gridMillis == 0) {
            return new SelValueGraphData(GraphData.graphData(begin, constant, end, constant));
        } else {
            long capacity = (end - begin) / gridMillis + 1;
            if (capacity >= 10_000) {
                throw new EvaluationException(args.getRange(1),
                        "grid is too small for selected interval, numPoints = " + capacity);
            }
            GraphDataArrayList builder = new GraphDataArrayList(Math.toIntExact(capacity));
            long beginAligned = ((begin + gridMillis - 1) / gridMillis) * gridMillis;
            for (long ts = beginAligned; ts < end; ts += gridMillis) {
                builder.add(ts, constant);
            }
            return new SelValueGraphData(builder.buildGraphDataFromSortedUniqueNoCheck());
        }
    }

    private static SelValue vectorLine(EvalContext ctx, ArgsList args) {
        return scalarLine(ctx, args).asSingleElementVector();
    }
}
