package ru.yandex.solomon.expression.expr.func;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.predicate.AggrPointPredicate;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.model.timeseries.FilteringAggrGraphDataIterator;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * Drop points if value is less than threshold
 * Example:
 * <pre>
 *     drop_below(line, 10) // drop points with values below 10
 * </pre>
 *
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class SelFnDropBelow implements SelFuncProvider {

    private static NamedGraphData calculate(NamedGraphData namedGraphData, double threshold) {
        AggrGraphDataArrayListOrView list = namedGraphData.getAggrGraphDataArrayList();
        if (list.isEmpty()) {
            return namedGraphData;
        }

        final AggrPointPredicate predicate;
        if (list.hasColumn(StockpileColumn.VALUE)) {
            predicate = (mask, point) -> {
                double value = point.getValueDivided();
                if (Double.isNaN(value) || Double.isInfinite(value)) {
                    return false;
                }

                return Double.compare(value, threshold) >= 0;
            };
        } else if (list.hasColumn(StockpileColumn.LONG_VALUE)) {
            long lthreshdold = Math.round(threshold);
            predicate = (mask, point) -> point.longValue >= lthreshdold;
        } else {
            throw new UnsupportedOperationException("Unsupported mask: " + list.columnSet());
        }

        AggrGraphDataArrayList result = new AggrGraphDataArrayList(list.columnSetMask(), list.length());
        result.addAllFrom(new FilteringAggrGraphDataIterator(list.iterator(), predicate));
        return namedGraphData.toBuilder()
            .setGraphData(result)
            .build();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("drop_below")
            .help("Drop points with value less then specified(exclusive)")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(arg("source").type(SelTypes.GRAPH_DATA), arg("threshold").type(SelTypes.DOUBLE))
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var source = args.get(0).castToGraphData().getNamedGraphData();
                var threshold = args.get(1).castToScalar().getValue();
                return new SelValueGraphData(calculate(source, threshold));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("drop_below")
            .help("Drop points with value less then specified(exclusive)")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(arg("source").type(SelTypes.GRAPH_DATA_VECTOR), arg("threshold").type(SelTypes.DOUBLE))
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var source = args.get(0).castToVector().valueArray();
                var threshold = args.get(1).castToScalar().getValue();
                var result = new SelValue[source.length];
                for (int index = 0; index < source.length; index++) {
                    var item = source[index].castToGraphData().getNamedGraphData();
                    result[index] = new SelValueGraphData(calculate(item, threshold));
                }

                return new SelValueVector(SelTypes.GRAPH_DATA, result);
            })
            .build());
    }
}
