package ru.yandex.solomon.expression.expr.func;

import java.util.List;
import java.util.function.BiFunction;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.base.Joiner;

import ru.yandex.solomon.expression.exceptions.InternalCompilerException;
import ru.yandex.solomon.expression.expr.IntrinsicsExternalizer;
import ru.yandex.solomon.expression.expr.SelExpr;
import ru.yandex.solomon.expression.expr.SelExprParam;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.labels.query.Selectors;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
public class SelFunc {
    private static final Pattern FUNCTION_NAME_REGEXP = Pattern.compile("^[a-z_0-9]+$");

    private final String name;
    private final String help;
    private final SelType returnType;
    private final List<SelFuncArgument> args;
    private final List<SelType> argsType;
    private final SelFuncCategory category;
    private final SelFunctionWithContext handler;
    private final boolean pure;
    private final boolean varArg;
    private final Predicate<SelVersion> supportedVersions;
    @Nullable
    private final SelIntervalHandler intervalHandler;
    @Nullable
    private final Externalizer externalizer;
    private final ExternalizerPostProcess externalizerPostProcess;

    private SelFunc(Builder builder) {
        this.name = requireNonNull(builder.name, "name");
        this.help = builder.help;
        this.returnType = requireNonNull(builder.returnType, "returnType");
        this.args = builder.args;
        this.argsType = builder.args.stream()
            .map(SelFuncArgument::getType)
            .collect(Collectors.toUnmodifiableList());
        this.category = requireNonNull(builder.category, "category");
        this.handler = builder.handler;
        this.pure = builder.pure;
        this.varArg = builder.varArg;
        this.intervalHandler = builder.intervalHandler;
        this.externalizer = builder.external;
        this.externalizerPostProcess = builder.externalizerPostProcess;
        this.supportedVersions = builder.supportedVersions;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public String getName() {
        return name;
    }

    public String getHelp() {
        return help;
    }

    public SelFuncCategory getCategory() {
        return category;
    }

    public SelType getReturnType() {
        return returnType;
    }

    public List<SelFuncArgument> getArgs() {
        return args;
    }

    public List<SelType> getArgsType() {
        return argsType;
    }

    public boolean isPure() {
        return pure;
    }

    public boolean isVarArg() {
        return varArg;
    }

    public SelFunctionWithContext getHandler() {
        return handler;
    }

    @Nullable
    public Externalizer getExternalizer() {
        return externalizer;
    }

    public ExternalizerPostProcess getExternalizerPostProcess() {
        return externalizerPostProcess;
    }

    @Nullable
    public SelIntervalHandler getIntervalHandler() {
        return intervalHandler;
    }

    public Predicate<SelVersion> getSupportedVersions() {
        return supportedVersions;
    }

    @Override
    public String toString() {
        String strArgs = Joiner.on(", ").join(argsType);
        if (varArg) {
            strArgs += "...";
        }
        return name + "(" + strArgs + "): " + returnType;
    }

    public interface Externalizer extends BiFunction<ArgsList, IntrinsicsExternalizer, Selectors> {
    }

    @FunctionalInterface
    public interface ExternalizerPostProcess {
        SelExpr apply(SelVersion version, ArgsList args, SelExprParam param);
    }

    public static class Builder {
        private String name;
        private String help;
        private SelType returnType;
        private List<SelFuncArgument> args = List.of();
        private SelFunctionWithContext handler;
        private Externalizer external;
        private ExternalizerPostProcess externalizerPostProcess = (ctx, argsList, param) -> param;
        private SelIntervalHandler intervalHandler;
        private SelFuncCategory category;
        private boolean pure = true;
        private boolean varArg = false;
        private Predicate<SelVersion> supportedVersions = SelVersion.MIN::since;

        private Builder() {
        }

        public Builder name(String name) {
            if (!FUNCTION_NAME_REGEXP.matcher(name).matches()) {
                throw new InternalCompilerException("Invalid function name: " + name);
            }
            this.name = name;
            return this;
        }

        public Builder category(SelFuncCategory category) {
            this.category = category;
            return this;
        }

        public Builder help(String help) {
            this.help = requireNonNull(help);
            return this;
        }

        public Builder returnType(SelType type) {
            this.returnType = requireNonNull(type);
            return this;
        }

        public Builder args(SelType... args) {
            var result = new SelFuncArgument[args.length];
            for (int index = 0; index < args.length; index++) {
                result[index] = SelFuncArgument.newBuilder()
                    .name("arg"+index)
                    .type(args[index])
                    .build();
            }

            return args(result);
        }

        public Builder args(SelFuncArgument... args) {
            this.args = List.of(args);
            return this;
        }

        public Builder args(SelFuncArgument.Builder... args) {
            this.args = Stream.of(args)
                .map(SelFuncArgument.Builder::build)
                .collect(Collectors.toUnmodifiableList());
            return this;
        }

        public Builder pure(boolean value) {
            this.pure = value;
            return this;
        }

        public Builder varArg(boolean value) {
            this.varArg = value;
            return this;
        }



        public Builder handler(SelFunctionHandler handler) {
            this.handler = (interval, selValues) -> handler.apply(selValues);
            return this;
        }

        public Builder handler(SelFunctionWithContext handler) {
            this.handler = handler;
            return this;
        }

        public Builder externalizer(Externalizer handler) {
            this.external = handler;
            return this;
        }

        public Builder externalizerPostProcess(ExternalizerPostProcess postProcess) {
            this.externalizerPostProcess = postProcess;
            return this;
        }

        /**
         * @param handler list argument can be null if not able calculate it during compile
         */
        public Builder interval(SelIntervalHandler handler) {
            this.intervalHandler = handler;
            return this;
        }

        public Builder supportedVersions(Predicate<SelVersion> supportedVersions) {
            this.supportedVersions = supportedVersions;
            return this;
        }

        public SelFunc build() {
            if (varArg && args.size() != 1) {
                throw new InternalCompilerException("VarArg function can be only with only one argument");
            }
            return new SelFunc(this);
        }
    }
}
