package ru.yandex.solomon.expression.expr.func;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Predicate;

import com.google.common.base.Preconditions;

import ru.yandex.solomon.expression.exceptions.InternalCompilerException;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.version.SelVersion;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
public class SelOp {
    private final String name;
    private final String operator;
    private final SelType returnType;
    private final List<SelType> args;
    private final SelFunctionWithContext handler;
    private final Predicate<SelVersion> supportedVersions;

    private SelOp(Builder builder) {
        this.name = requireNonNull(builder.name, "name");
        this.operator = requireNonNull(builder.operator, "operator");
        this.returnType = requireNonNull(builder.returnType, "returnType");
        Preconditions.checkArgument(builder.args.size() > 0);
        this.args = builder.args;
        this.handler = requireNonNull(builder.handler, "handler");
        this.supportedVersions = builder.supportedVersions;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static String format(String op, List args) {
        if (args.size() == 1) {
            return op + args.get(0);
        }
        return args.get(0) + " " + op + " " + args.get(1);
    }

    public String getName() {
        return name;
    }

    public String getOperator() {
        return operator;
    }

    public SelType getReturnType() {
        return returnType;
    }

    public List<SelType> getArgs() {
        return args;
    }

    public SelFunctionWithContext getHandler() {
        return handler;
    }

    public Predicate<SelVersion> getSupportedVersions() {
        return supportedVersions;
    }

    @Override
    public String toString() {
        return format(operator, args);
    }

    public static class Builder {
        private String name;
        private String operator;
        private SelType returnType;
        private List<SelType> args = List.of();
        private SelFunctionWithContext handler;
        private Predicate<SelVersion> supportedVersions = SelVersion.MIN::since;

        private Builder() {
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder operator(String operator) {
            this.operator = operator;
            return this;
        }

        public Builder returnType(SelType type) {
            this.returnType = requireNonNull(type);
            return this;
        }

        public Builder args(SelType... args) {
            if (args.length > 2) {
                throw new InternalCompilerException("Operator able to apply on on one or two argument, but was:" + Arrays.toString(args));
            }
            this.args = List.of(args);
            return this;
        }

        public Builder handler(SelFunctionHandler handler) {
            this.handler = (ignoreCtx, args) -> handler.apply(args);
            return this;
        }

        public Builder handler(SelFunctionWithContext handler) {
            this.handler = handler;
            return this;
        }

        public Builder supportedVersions(Predicate<SelVersion> supportedVersions) {
            this.supportedVersions = supportedVersions;
            return this;
        }

        public SelOp build() {
            return new SelOp(this);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SelOp selOp = (SelOp) o;
        return name.equals(selOp.name) &&
            operator.equals(selOp.operator) &&
            returnType.equals(selOp.returnType) &&
            args.equals(selOp.args) &&
            handler.equals(selOp.handler);
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, operator, returnType, args, handler);
    }
}
