package ru.yandex.solomon.expression.expr.func;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.AstOp;
import ru.yandex.solomon.expression.exceptions.CompilerException;
import ru.yandex.solomon.expression.exceptions.InternalCompilerException;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.version.SelVersion;

/**
 * @author Vladimir Gordiychuk
 */
public class SelOpRegistry {
    private final EnumMap<SelVersion, Table<String, List<SelType>, SelOp>> registered;

    public SelOpRegistry() {
        this.registered = new EnumMap<>(Arrays.stream(SelVersion.values())
                .collect(Collectors.toMap(Function.identity(), version -> HashBasedTable.create())));
    }

    public void add(SelOp.Builder builder) {
        add(builder.build());
    }

    public void add(SelOp op) {
        Arrays.stream(SelVersion.values())
                .filter(op.getSupportedVersions())
                .forEach(version -> add(version, op));
    }

    public void add(SelVersion version, SelOp op) {
        var prev = registered.get(version).get(op.getOperator(), op.getArgs());
        if (prev != null) {
            throw new InternalCompilerException(prev + " already registered");
        }
        registered.get(version).put(op.getOperator(), op.getArgs(), op);
    }

    public void add(SelOp... ops) {
        for (var op : ops) {
            add(op);
        }
    }

    public void add(SelOpProvider provider) {
        provider.provide(this);
    }

    public SelOp get(SelVersion version, AstOp op, List<SelType> args) {
        var overloads = registered.get(version).row(op.asString());
        if (overloads.isEmpty()) {
            throw new CompilerException(op.getRange(), "Unknown operator: " + op.asString());
        }

        return resolveOperator(op, overloads, args);
    }

    @VisibleForTesting
    public SelOp get(SelVersion version, String op, List<SelType> args) {
        return get(version, new AstOp(PositionRange.UNKNOWN, op), args);
    }

    private static SelOp resolveOperator(AstOp operator, Map<List<SelType>, SelOp> overloads, List<SelType> args) {
        var op = overloads.get(args);
        if (op == null) {
            String was = SelOp.format(operator.asString(), args);
            String expected = "[" + Joiner.on(", ").join(overloads.values()) + "]";
            String message = "Not valid operator arguments, was " + was + ", but expected " + expected;
            throw new CompilerException(operator.getRange(), message);
        }

        return op;
    }
}
