package ru.yandex.solomon.expression.expr.func.aggr;

import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.exceptions.TooManyLinesInAggregation;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncArgument;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueDouble;
import ru.yandex.solomon.math.doubles.DoubleAggregateFunctions;
import ru.yandex.solomon.math.doubles.DoubleTimeSeriesFactory;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;
import static ru.yandex.solomon.expression.expr.func.analytical.histogram.SelFnHistogramPercentile.getPercentileLevel;

/**
 * @author Ivan Tsybulin
 *
 * Usage:
 * <pre>
 *     let data = {...};
 *     alarm_if(percentile(90, data) > 10K); // don't alarm on single outbursts
 * </pre>
 */
@ParametersAreNonnullByDefault
public class SelFnPercentile implements SelFuncProvider {
    private static String name() {
        return "percentile";
    }

    private static String help() {
        return "Compute percentile of given values";
    }

    private static double evalSingleLine(NamedGraphData source, double percentileLevel) {
        var view = DoubleTimeSeriesFactory.fromGraphData(source.getDataType(), source.getAggrGraphDataArrayList());
        return DoubleAggregateFunctions.percentile(view.getValueView(), percentileLevel);
    }

    private static double evalMultipleLines(PositionRange range, SelValue[] source, double percentileLevel) {
        if (source.length != 1) {
            throw new TooManyLinesInAggregation(range, name(), Stream.of(source)
                    .map(v -> v.castToGraphData().getNamedGraphData())
                    .collect(Collectors.toList())
            );
        }

        return evalSingleLine(source[0].castToGraphData().getNamedGraphData(), percentileLevel);
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        SelFuncArgument level = arg("level")
                .type(SelTypes.DOUBLE)
                .help("Percentile level, should be in [0, 100] range")
                .build();
        SelFuncArgument values = arg("values")
                .type(SelTypes.DOUBLE_VECTOR)
                .help("Sample values")
                .build();
        SelFuncArgument line = arg("series")
                .type(SelTypes.GRAPH_DATA)
                .help("Sample values")
                .build();
        SelFuncArgument lines = arg("series")
                .type(SelTypes.GRAPH_DATA_VECTOR)
                .help("Sample values")
                .build();
        registry.add(SelFunc.newBuilder()
            .name(name())
            .help(help())
            .category(SelFuncCategory.AGGREGATION)
            .args(level, values)
            .returnType(SelTypes.DOUBLE)
            .handler(args -> {
                double percentileLevel = getPercentileLevel(args.getWithRange(0));
                var source = new DoubleArrayView(args.get(1).castToVector().doubleArray());
                return new SelValueDouble(DoubleAggregateFunctions.percentile(source, percentileLevel));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name(name())
            .help(help())
            .category(SelFuncCategory.AGGREGATION)
            .args(level, line)
            .returnType(SelTypes.DOUBLE)
            .handler(args -> {
                double percentileLevel = getPercentileLevel(args.getWithRange(0));
                NamedGraphData source = args.get(1).castToGraphData().getNamedGraphData();
                return new SelValueDouble(evalSingleLine(source, percentileLevel));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name(name())
            .help(help())
            .category(SelFuncCategory.AGGREGATION)
            .args(level, lines)
            .returnType(SelTypes.DOUBLE)
            .handler(args -> {
                double percentileLevel = getPercentileLevel(args.getWithRange(0));
                var source = args.get(1).castToVector().valueArray();
                return new SelValueDouble(evalMultipleLines(args.getRange(1), source, percentileLevel));
            })
            .build());
    }
}
