package ru.yandex.solomon.expression.expr.func.analytical;

import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.math.stat.ASAP;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.util.time.Interval;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * Apply moving average by auto choose window size
 * Example usage:
 * <pre>{@code
 *   asap(myMetric{host=cluster});
 * }</pre>
 *
 * @author Vladimir Gordiychuk
 * @see ASAP
 */
@ParametersAreNonnullByDefault
public class SelFnAsap implements SelFuncProvider {
    public static final String NAME = "asap";

    private static NamedGraphData calculate(NamedGraphData value) {
        GraphData result = ASAP.smooth(value.getGraphData());
        return value.toBuilder()
            .setGraphData(result)
            .build();
    }

    private static Interval extendWindowIteration(Interval source) {
        var train = source.duration().dividedBy(5);
        return new Interval(source.getBegin().minus(train), source.getEnd());
    }

    private static Interval extendWindow(Interval source) {
        return extendWindowIteration(extendWindowIteration(source));
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name(NAME)
            .category(SelFuncCategory.TRANSFORMATION)
            .help("Compute a moving average with a span automatically selected to remove as " +
                "much noise as possible while preserving important trends.")
            .args(arg("source").type(SelTypes.GRAPH_DATA))
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var source = args.get(0).castToGraphData().getNamedGraphData();
                return new SelValueGraphData(calculate(source));
            })
            .interval((interval, args) -> extendWindow(interval))
            .build());

        registry.add(SelFunc.newBuilder()
            .name(NAME)
            .category(SelFuncCategory.TRANSFORMATION)
            .help("Compute a moving average with a span automatically selected to remove as " +
                "much noise as possible while preserving important trends.")
            .args(arg("source").type(SelTypes.GRAPH_DATA_VECTOR))
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var source = args.get(0).castToVector().valueArray();
                var result = Stream.of(source)
                    .parallel()
                    .map(value -> new SelValueGraphData(calculate(value.castToGraphData().getNamedGraphData())))
                    .toArray(SelValue[]::new);
                return new SelValueVector(SelTypes.GRAPH_DATA, result);
            })
            .interval((interval, args) -> extendWindow(interval))
            .build());
    }
}
