package ru.yandex.solomon.expression.expr.func.analytical;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.value.SelValues;
import ru.yandex.solomon.math.GraphDataMath;
import ru.yandex.solomon.model.timeseries.GraphData;

/**
 * Drop data from timeseries using an extra series as a drop indicator
 * Example usage:
 * <pre>{@code
 *   let source = myMetric{host=*};
 *   let alerts = {name=breakdown};
 *   // Exclude points from source pushed at breakdowns
 *   let regularData = drop_if(alerts, source);
 * }</pre>
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnDropIf implements SelFuncProvider {

    private static SelValue dropIf(GraphData condition, SelValueWithRange source) {
        return SelValues.mapToGraphData(source, gd -> GraphDataMath.dropIf(condition, gd));
    }

    private static SelValue calculate(ArgsList args) {
        SelValue conditionParam = SelValues.getSingleGraphData(args.getWithRange(0));
        return dropIf(conditionParam.castToGraphData().getGraphData(), args.getWithRange(1));
    }

    private static SelFunc function(SelType... args) {
        return SelFunc.newBuilder()
            .name("drop_if")
            .help("Drop data from timeseries using an extra series as a drop indicator")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(args)
            .returnType(args[1])
            .handler(SelFnDropIf::calculate)
            .build();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(function(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA));
        registry.add(function(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA_VECTOR));
        registry.add(function(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA));
        registry.add(function(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA_VECTOR));
    }
}
