package ru.yandex.solomon.expression.expr.func.analytical;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.value.SelValues;
import ru.yandex.solomon.math.GraphDataMath;

/**
 * Return everything except the last part of timeseries by count points or duration
 * Example usage:
 * <pre>{@code
 *   let source = myMetric{host=cluster};
 *   // drop last 42 points from source metric
 *   let exceptLastNPoints = drop_tail(source, 42);
 *   // drop last 10 minutes from source metric
 *   let exceptTenMin = drop_tail(source, 10m);
 * }</pre>
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnDropTail implements SelFuncProvider {

    private static SelValue tailByPoints(SelValueWithRange source, int points) {
        return SelValues.mapToGraphData(source, gd -> GraphDataMath.dropTail(gd, points));
    }

    private static SelValue tailByDuration(SelValueWithRange source, Duration duration) {
        return SelValues.mapToGraphData(source, gd -> GraphDataMath.dropTail(gd, duration));
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("drop_tail")
            .help("Return everything except the last part of timeseries by count points")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(SelTypes.GRAPH_DATA, SelTypes.DOUBLE)
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var source = args.getWithRange(0);
                var count = (int) args.get(1).castToScalar().getValue();

                return tailByPoints(source, count);
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("drop_tail")
            .help("Return everything except the last part of timeseries by count points")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var source = args.getWithRange(0);
                var count = (int) args.get(1).castToScalar().getValue();

                return tailByPoints(source, count);
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("drop_tail")
            .help("Return everything except the last part of timeseries by duration")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(SelTypes.GRAPH_DATA, SelTypes.DURATION)
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var source = args.getWithRange(0);
                var duration = args.get(1).castToDuration().getDuration();

                return tailByDuration(source, duration);
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("drop_tail")
            .help("Return everything except the last part of timeseries by duration")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(SelTypes.GRAPH_DATA_VECTOR, SelTypes.DURATION)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var source = args.getWithRange(0);
                var duration = args.get(1).castToDuration().getDuration();

                return tailByDuration(source, duration);
            })
            .build());
    }
}
