package ru.yandex.solomon.expression.expr.func.analytical;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.parser.DurationIntervalsUnion;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.value.SelValues;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.GraphDataArrayList;

/**
 * Filter data by daytime
 * Example usage:
 * <pre>{@code
 *   let source = myMetric{host=cluster};
 *   let daytime = filter_by_time(source, '[6h-18h]');
 *   let night = filter_by_time(source, '[0h-4h] + [20h-24h]');
 * }</pre>
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnFilterByTime implements SelFuncProvider {
    private final static long MILLIS_PER_DAY = TimeUnit.DAYS.toMillis(1);

    private static GraphData filterGraphData(GraphData data, DurationIntervalsUnion filter) {
        GraphDataArrayList list = new GraphDataArrayList(data.length());
        data.visit((tsMillis, value) -> {
            long millisOfDay = Math.floorMod(tsMillis, MILLIS_PER_DAY);
            if (filter.contains(millisOfDay))
                list.add(tsMillis, value);
        });
        return list.buildGraphData();
    }

    private static SelValue filter(SelValueWithRange data, DurationIntervalsUnion condition) {
        return SelValues.mapToGraphData(data, gd -> filterGraphData(gd, condition));
    }

    private static SelValue calculate(ArgsList args) {
        SelValueWithRange data = args.getWithRange(0);
        String intervalsStr = args.get(1).castToString().getValue();
        DurationIntervalsUnion diu = DurationIntervalsUnion.parse(intervalsStr);
        return filter(data, diu);
    }

    private static SelFunc function(SelType... args) {
        return SelFunc.newBuilder()
            .name("filter_by_time")
            .help("Filter data by daytime")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(args)
            .returnType(args[0])
            .handler(SelFnFilterByTime::calculate)
            .build();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(function(SelTypes.GRAPH_DATA, SelTypes.STRING));
        registry.add(function(SelTypes.GRAPH_DATA_VECTOR, SelTypes.STRING));
    }
}
