package ru.yandex.solomon.expression.expr.func.analytical;

import java.time.Duration;
import java.util.LongSummaryStatistics;
import java.util.Objects;
import java.util.stream.LongStream;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.operation.map.OperationDownsampling;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling.FillOption;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.solomon.util.time.Interval;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * <p>Group time-series by user-specified time interval. As a result of group function all time-series will have common
 * time line.
 * <p>Example usage {@code group_by_time(15s, 'max', load('host=*'))}
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelFnGroupByTime implements SelFuncProvider {
    private static SelValueVector getVectorGraphData(ArgsList args) {
        SelValue selValue = args.get(2);
        if (selValue.type().isVector()) {
            return selValue.castToVector();
        }

        return new SelValueVector(SelTypes.GRAPH_DATA, new SelValue[]{selValue.castToGraphData()});
    }

    private static Duration getDuration(ArgsList args) {
        Duration duration = args.get(0).castToDuration().getDuration();
        if (duration.getSeconds() < 1) {
            throw new EvaluationException(args.getRange(0), "Min available interval for group points 1s, " +
                "but was " + DurationUtils.toStringUpToMillis(duration) + "s");
        }

        // TODO hack to prevent not valid grouping in grafana
        if (duration.getSeconds() < 15) {
            duration = Duration.ofSeconds(15);
        }

        return duration;
    }

    private static Aggregation getAggregation(ArgsList args) {
        String name = args.get(1).castToString().getValue();
        for (Aggregation aggr : Aggregation.values()) {
            if (aggr.name().equalsIgnoreCase(name)) {
                return aggr;
            }
        }

        throw new EvaluationException(args.getRange(1), "Unsupported aggregation: " + name);
    }

    @Nullable
    private static Interval getInterval(SelValue[] values) {
        LongSummaryStatistics summary = Stream.of(values)
            .map(selValue -> selValue.castToGraphData().getNamedGraphData().getAggrGraphDataArrayList())
            .filter(ts -> !ts.isEmpty())
            .flatMapToLong(ts -> LongStream.of(ts.getTsMillis(0), ts.getTsMillis(ts.length() - 1)))
            .summaryStatistics();

        if (summary.getCount() == 0) {
            return null;
        }

        return Interval.millis(summary.getMin(), summary.getMax());
    }

    private static SelValueVector calculate(ArgsList args) {
        var opts = ru.yandex.solomon.math.protobuf.OperationDownsampling.newBuilder()
            .setGridMillis(getDuration(args).toMillis())
            .setAggregation(getAggregation(args))
            .setFillOption(FillOption.NULL)
            .build();

        SelValueVector vector = getVectorGraphData(args);
        SelValue[] values = vector.valueArray();
        if (vector.length() == 0) {
            return vector;
        }

        Interval interval = getInterval(values);
        if (interval == null) {
            return vector;
        }

        var operation = new OperationDownsampling<Labels>(interval, opts);
        SelValue[] result = new SelValue[values.length];
        for (int index = 0; index < result.length; index++) {
            NamedGraphData gd = values[index].castToGraphData().getNamedGraphData();
            Labels key = gd.getLabels();
            MetricType type = gd.getDataType();
            AggrGraphDataArrayListOrView ts = gd.getAggrGraphDataArrayList();

            AggrGraphDataIterable timeseries =
                Objects.requireNonNull(operation.apply(new Metric<>(key, type, ts)).getTimeseries());
            AggrGraphDataArrayList tsResult = AggrGraphDataArrayList.of(timeseries);

            NamedGraphData newNamedGraphData = gd.toBuilder()
                .setGraphData(type, tsResult)
                .build();

            result[index] = new SelValueGraphData(newNamedGraphData);
        }

        return new SelValueVector(SelTypes.GRAPH_DATA, result);
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        String[] availableAggregations = Stream.of(Aggregation.values())
            .filter(aggregation -> aggregation != Aggregation.DEFAULT_AGGREGATION && aggregation != Aggregation.UNRECOGNIZED)
            .map(aggregation -> aggregation.name().toLowerCase())
            .toArray(String[]::new);

        registry.add(SelFunc.newBuilder()
            .name("group_by_time")
            .help("Group points into specified window by aggregation function")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("window").type(SelTypes.DURATION).help("group points to apply aggregation function"),
                arg("aggregation").type(SelTypes.STRING).help("function applied on points").availableValues(availableAggregations),
                arg("source").type(SelTypes.GRAPH_DATA)
            )
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> calculate(args).item(0))
            .build());

        registry.add(SelFunc.newBuilder()
            .name("group_by_time")
            .help("Group points into specified window by aggregation function")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("window").type(SelTypes.DURATION).help("group points to apply aggregation function"),
                arg("aggregation").type(SelTypes.STRING).help("function applied on points").availableValues(availableAggregations),
                arg("source").type(SelTypes.GRAPH_DATA_VECTOR)
            )
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(SelFnGroupByTime::calculate)
            .build());
    }
}
