package ru.yandex.solomon.expression.expr.func.analytical;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.value.SelValues;
import ru.yandex.solomon.math.GraphDataMath;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * Return first part of timeseries by count points or duration
 * Example usage:
 * <pre>{@code
 *   let source = myMetric{host=cluster};
 *   // return first 42 points from source metric
 *   let firstNPoints = head(source, 42);
 *   // return first 10 minutes from source metric
 *   let tenMin = head(source, 10m);
 * }</pre>
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelFnHead implements SelFuncProvider {
    private static SelValue headByPoints(SelValueWithRange source, int points) {
        return SelValues.mapToGraphData(source, gd -> GraphDataMath.head(gd, points));
    }

    private static SelValue headByDuration(SelValueWithRange source, Duration duration) {
        return SelValues.mapToGraphData(source, gd -> GraphDataMath.head(gd, duration));
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("head")
            .help("Return first part of timeseries by count points")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA),
                arg("count").type(SelTypes.DOUBLE).help("count points to extract or less if not enough"))
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var source = args.getWithRange(0);
                var count = (int) args.get(1).castToScalar().getValue();

                return headByPoints(source, count);
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("head")
            .help("Return first part of timeseries by count points")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA_VECTOR),
                arg("count").type(SelTypes.DOUBLE).help("count points to extract or less if not enough"))
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var source = args.getWithRange(0);
                var count = (int) args.get(1).castToScalar().getValue();

                return headByPoints(source, count);
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("head")
            .help("Return first part of timeseries by duration")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA),
                arg("window").type(SelTypes.DURATION))
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var source = args.getWithRange(0);
                var duration = args.get(1).castToDuration().getDuration();

                return headByDuration(source, duration);
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("head")
            .help("Return first part of timeseries by duration")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA_VECTOR),
                arg("window").type(SelTypes.DURATION))
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var source = args.getWithRange(0);
                var duration = args.get(1).castToDuration().getDuration();

                return headByDuration(source, duration);
            })
            .build());
    }
}
