package ru.yandex.solomon.expression.expr.func.analytical;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.value.SelValues;
import ru.yandex.solomon.math.stat.KronosTrend;

/**
 * Return predicted variance by Kronos algorithm.
 * Signature:
 * <pre>
 * kronos_adjusted(train: Timeseries*, test: Timeseries*, dailyBins: Scalar, smoothWindow: Duration,
 *                 timezone: Duration, dropFraction: Scalar, minBucketPoints: Scalar,
 *                 relMinVar: Scalar, absMinVar: Scalar) -> Timeseries*
 * </pre>
 * Here Timeseries* means Timeseries or Vector of Timeseries.
 * Example usage:
 * <pre>{@code
 *   let source = myMetric{host=cluster};
 *   let fitData = drop_tail(source, 10m);
 *   let predictData = tail(source, 10m);
 *   let adjusted = kronos_adjusted(fitData, predictData, 24 * 60, 5m, +3h, 0.1, 10, 0.01, 0);
 * }</pre>
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnKronosAdjusted implements SelFuncProvider {
    private static SelValue variance(
            SelValueWithRange fit,
            SelValueWithRange predict,
            int intervalsPerDay,
            Duration smoothWindow,
            Duration timezoneOffset,
            double dropFraction,
            int minBucketPoints,
            double relMinVar,
            double absMinVar)
    {
        return SelValues.mapToGraphData(fit, predict, (fitGd, predictGd) -> {
            KronosTrend kt = new KronosTrend(intervalsPerDay, smoothWindow, dropFraction, timezoneOffset, minBucketPoints);
            kt.fit(fitGd);
            return kt.predictAdjusted(predictGd, relMinVar, absMinVar);
        });
    }

    private static SelValue calculate(ArgsList args) {
        SelValueWithRange fit = args.getWithRange(0);
        SelValueWithRange predict = args.getWithRange(1);
        int intervalsPerDay = (int) args.get(2).castToScalar().getValue();
        Duration smoothWindow = args.get(3).castToDuration().getDuration();
        Duration timezoneOffset = args.get(4).castToDuration().getDuration();
        double dropFraction = args.get(5).castToScalar().getValue();
        int minBucketPoints = (int) args.get(6).castToScalar().getValue();
        double relMinVar = args.get(7).castToScalar().getValue();
        double absMinVar = args.get(8).castToScalar().getValue();
        return variance(fit, predict, intervalsPerDay, smoothWindow, timezoneOffset, dropFraction,
            minBucketPoints, relMinVar, absMinVar);
    }

    private static SelFunc function(SelType... args) {
        var returnType = args[0] == SelTypes.GRAPH_DATA_VECTOR && args[1] == SelTypes.GRAPH_DATA_VECTOR
            ? SelTypes.GRAPH_DATA_VECTOR
            : SelTypes.GRAPH_DATA;
        return SelFunc.newBuilder()
            .name("kronos_adjusted")
            .help("Return predicted variance by Kronos algorithm")
            .category(SelFuncCategory.PREDICT)
            .args(args)
            .returnType(returnType)
            .handler(SelFnKronosAdjusted::calculate)
            .build();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(function(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA, SelTypes.DOUBLE, SelTypes.DURATION, SelTypes.DURATION, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE));
        registry.add(function(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE, SelTypes.DURATION, SelTypes.DURATION, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE));
        registry.add(function(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA, SelTypes.DOUBLE, SelTypes.DURATION, SelTypes.DURATION, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE));
        registry.add(function(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE, SelTypes.DURATION, SelTypes.DURATION, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE));
    }
}
