package ru.yandex.solomon.expression.expr.func.analytical;

import java.time.Duration;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.exceptions.PreparingException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.math.stat.MovingAverage;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.util.time.Interval;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * <p>Simple moving average (SMA) is the unweighted mean of the previous N data.
 * <p>Example usage {@code moving_avg(graphData, '1h')}
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelFnMovingAvg implements SelFuncProvider {
    public static final String NAME = "moving_avg";

    private static NamedGraphData calculate(NamedGraphData namedGraphData, Duration duration) {
        GraphData movingAvg = MovingAverage.simple(namedGraphData.getGraphData(), duration);
        return namedGraphData.toBuilder()
            .setGraphData(movingAvg)
            .build();
    }

    private static Duration getWindow(ArgsList args) {
        return Optional.ofNullable(args.get(1))
            .map(value -> value.castToDuration().getDuration())
            .orElseThrow(() -> new PreparingException(args.getRange(1), "Unable to parse duration argument at compile time for moving_avg"));
    }

    private static Interval interval(Interval interval, ArgsList args) {
        var window = getWindow(args);
        return new Interval(interval.getBegin().minus(window), interval.getEnd());
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name(NAME)
            .help("Simple moving average (SMA) is the unweighted mean of the previous N data.")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA),
                arg("window").type(SelTypes.DURATION))
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var source = args.get(0).castToGraphData().getNamedGraphData();
                var window = args.get(1).castToDuration().getDuration();
                return new SelValueGraphData(calculate(source, window));
            })
            .interval(SelFnMovingAvg::interval)
            .build());

        registry.add(SelFunc.newBuilder()
            .name(NAME)
            .help("Simple moving average (SMA) is the unweighted mean of the previous N data.")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA_VECTOR),
                arg("window").type(SelTypes.DURATION))
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var source = args.get(0).castToVector().valueArray();
                var window = args.get(1).castToDuration().getDuration();

                var result = new SelValue[source.length];
                for (int index = 0; index < source.length; index++) {
                    var item = source[index].castToGraphData().getNamedGraphData();
                    result[index] = new SelValueGraphData(calculate(item, window));
                }

                return new SelValueVector(SelTypes.GRAPH_DATA, result);
            })
            .interval(SelFnMovingAvg::interval)
            .build());
    }
}
