package ru.yandex.solomon.expression.expr.func.analytical;

import java.time.Duration;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.exceptions.PreparingException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueDuration;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.math.stat.MovingPercentile;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.util.time.Interval;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * Example usage {@code moving_percentile(graphData, '1h', 99.9)}
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelFnMovingPercentile implements SelFuncProvider {
    public static final String NAME = "moving_percentile";

    private static Duration getWindow(ArgsList args) {
        return Optional.ofNullable(args.get(1))
            .map(SelValue::castToDuration)
            .map(SelValueDuration::getDuration)
            .orElseThrow(() -> new PreparingException(args.getRange(1), "Specified interval is not valid"));
    }

    private static Interval interval(Interval interval, ArgsList args) {
        var window = getWindow(args);
        return new Interval(interval.getBegin().minus(window), interval.getEnd());
    }

    private NamedGraphData calculate(NamedGraphData namedGraphData, Duration window, double rank) {
        GraphData result = MovingPercentile.simple(namedGraphData.getGraphData(), window, rank);
        return namedGraphData.toBuilder()
            .setGraphData(result)
            .build();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name(NAME)
            .help("Simple moving percentile is the percentile of the previous N data inside interval.")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA),
                arg("window").type(SelTypes.DURATION).help("window define points range on witch will be calculate percentile"),
                arg("rank").type(SelTypes.DOUBLE).help("value of percentile, should be in interval from 0 to 100, for example 99.9")
            )
            .returnType(SelTypes.GRAPH_DATA)
            .interval(SelFnMovingPercentile::interval)
            .handler(args -> {
                var source = args.get(0).castToGraphData().getNamedGraphData();
                var window = args.get(1).castToDuration().getDuration();
                var rank = args.get(2).castToScalar().getValue();
                return new SelValueGraphData(calculate(source, window, rank));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name(NAME)
            .help("Simple moving percentile is the percentile of the previous N data inside interval.")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA_VECTOR),
                arg("window").type(SelTypes.DURATION).help("window define points range on witch will be calculate percentile"),
                arg("rank").type(SelTypes.DOUBLE).help("value of percentile, should be in interval from 0 to 100, for example 99.9")
            )
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .interval(SelFnMovingPercentile::interval)
            .handler(args -> {
                var source = args.get(0).castToVector().valueArray();
                var window = args.get(1).castToDuration().getDuration();
                var rank = args.get(2).castToScalar().getValue();

                var result = new SelValue[source.length];
                for (int index = 0; index < source.length; index++) {
                    var item = source[index].castToGraphData().getNamedGraphData();
                    result[index] = new SelValueGraphData(calculate(item, window, rank));
                }

                return new SelValueVector(SelTypes.GRAPH_DATA, result);
            })
            .build());
    }
}
