package ru.yandex.solomon.expression.expr.func.analytical;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.expr.func.LabelsUtil;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.math.stat.OverLinePercentile;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.Timeline;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;


/**
 * <p>Apply percentile as aggregate function for values at the same time point over all lines
 * <p>
 * <p>Example usage {@code percentile_group_lines(99.9, group_by_time(15s, 'max', graphDataVector))}
 *
 * @author Vladimir Gordiychuk
 * @see SelFnGroupLines
 */
@ParametersAreNonnullByDefault
public class SelFnPercentileGroupLines implements SelFuncProvider {
    private static List<NamedGraphData> getTargetLines(ArgsList params) {
        return Stream.of(params.get(1).castToVector().valueArray())
            .map(v -> v.castToGraphData().getNamedGraphData())
            .collect(Collectors.toList());
    }

    private static NamedGraphData calculate(
        List<NamedGraphData> lines,
        Timeline timeline,
        double percentile)
    {
        Labels commonLabels = LabelsUtil.getCommonLabels(lines);

        List<GraphData> graphLines = lines.stream()
            .map(NamedGraphData::getGraphData)
            .collect(Collectors.toList());

        GraphData resultLine = OverLinePercentile.percentileOnParticularTime(graphLines, timeline, percentile);
        String alias = String.format("p%s", percentile);
        return NamedGraphData.newBuilder()
            .setGraphData(resultLine)
            .setAlias(alias)
            .setLabels(commonLabels)
            .build();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        provide("percentile_group_lines", registry, SelFuncCategory.DEPRECATED);
        provide("series_percentile", registry, SelFuncCategory.COMBINE);
    }

    public void provide(String name, SelFuncRegistry registry, SelFuncCategory category) {
        var rankArg = arg("rank").type(SelTypes.DOUBLE).build();
        var ranksArg = arg("ranks").type(SelTypes.DOUBLE_VECTOR).build();
        var sourceArg = arg("source").type(SelTypes.GRAPH_DATA_VECTOR).build();

        registry.add(SelFunc.newBuilder()
            .name(name)
            .help("Apply percentile as aggregate function for values at the same time point over all lines")
            .category(category)
            .args(rankArg, sourceArg)
            .supportedVersions(SelVersion.GROUP_LINES_RETURN_VECTOR_2::before)
            .returnType(SelTypes.GRAPH_DATA)
            .handler(SelFnPercentileGroupLines::singleRank)
            .build());

        registry.add(SelFunc.newBuilder()
            .name(name)
            .help("Apply percentile as aggregate function for values at the same time point over all lines")
            .category(category)
            .args(rankArg, sourceArg)
            .supportedVersions(SelVersion.GROUP_LINES_RETURN_VECTOR_2::since)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> singleRank(args).asSingleElementVector())
            .build());

        registry.add(SelFunc.newBuilder()
            .name(name)
            .help("Apply percentile as aggregate function for values at the same time point over all lines")
            .category(category)
            .args(ranksArg, sourceArg)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(SelFnPercentileGroupLines::multipleRanks)
            .build());
    }

    private static SelValue multipleRanks(ArgsList args) {
        var percentiles = args.get(0).castToVector().doubleArray();
        var source = getTargetLines(args);
        var timeline = Timeline.union(source.stream()
                .map(namedGraphData -> namedGraphData.getGraphData().getTimeline())
                .collect(Collectors.toList()));
        var result = new SelValue[percentiles.length];
        for (int index = 0; index < percentiles.length; index++) {
            var percentile = percentiles[index];
            result[index] = new SelValueGraphData(calculate(source, timeline, percentile));
        }
        return new SelValueVector(SelTypes.GRAPH_DATA, result);
    }

    private static SelValue singleRank(ArgsList args) {
        var percentile = args.get(0).castToScalar().getValue();
        var source = getTargetLines(args);
        var timeline = Timeline.union(source.stream()
                .map(namedGraphData -> namedGraphData.getGraphData().getTimeline())
                .collect(Collectors.toList()));
        return new SelValueGraphData(calculate(source, timeline, percentile));
    }
}
