package ru.yandex.solomon.expression.expr.func.analytical;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * Example usage {@code replace_nan(group_by_time(15s, 'max', load('host=*')), 0)}
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelFnReplaceNan implements SelFuncProvider {
    private static NamedGraphData calculate(NamedGraphData named, double replacement) {
        var source = named.getAggrGraphDataArrayList();
        var result = new AggrGraphDataArrayList(source.columnSetMask(), source.length());
        var point = RecyclableAggrPoint.newInstance();
        try {
            var it = source.iterator();
            while (it.next(point)) {
                if (Double.isNaN(point.getValueDivided())) {
                    point.setValue(replacement);
                }
                result.addRecord(point);
            }
        } finally {
            point.recycle();
        }
        return named.toBuilder()
            .setGraphData(result)
            .build();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("replace_nan")
            .help("Instead of NaN value set specified value")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA),
                arg("replace").type(SelTypes.DOUBLE).help("value used instead of NaN"))
            .returnType(SelTypes.GRAPH_DATA)
            .handler(args -> {
                var source = args.get(0).castToGraphData().getNamedGraphData();
                var replace = args.get(1).castToScalar().getValue();

                return new SelValueGraphData(calculate(source, replace));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("replace_nan")
            .help("Instead of NaN value set specified value")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA_VECTOR),
                arg("replace").type(SelTypes.DOUBLE).help("value used instead of NaN"))
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(args -> {
                var source = args.get(0).castToVector().valueArray();
                var replace = args.get(1).castToScalar().getValue();

                var result = new SelValue[source.length];
                for (int index = 0; index < source.length; index++) {
                    var item = source[index].castToGraphData().getNamedGraphData();
                    result[index] = new SelValueGraphData(calculate(item, replace));
                }

                return new SelValueVector(SelTypes.GRAPH_DATA, result);
            })
            .build());
    }
}
