package ru.yandex.solomon.expression.expr.func.analytical;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.value.SelValues;
import ru.yandex.solomon.math.stat.CalendarSplitter;
import ru.yandex.solomon.math.stat.DailyProfile;
import ru.yandex.solomon.math.stat.SeasonalTrend;

/**
 * Return adjusted (that is (x - mean) / std) seasonal data
 * with given profile specified by intervals per day count
 * and day bucketing scheme
 * The standart deviation std is computed as
 * sqrt(max(var, relMinVar * mean**2 + absMinVar))
 * to avoid division by zero
 * <p>
 * Example usage:
 * <pre>{@code
 *   let source = myMetric{host=cluster};
 *   let fitData = drop_tail(source, 10m);
 *   let normalizedNoise = seasonal_adjusted(fitData, source, 12, 'daily', +3h, 0.1);
 *   let normalizedNoiseRobust = seasonal_adjusted(fitData, source, 12, 'daily', +3h, 0.1, 0, 1e-3);
 * }</pre>
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnSeasonalAdjusted implements SelFuncProvider {
    private static SelValue adjusted(
            SelValueWithRange fit,
            SelValueWithRange predict,
            int intervalsPerDay,
            DailyProfile profile,
            Duration timezoneOffset,
            double dropFraction,
            double relMinVar,
            double absMinVar)
    {
        CalendarSplitter calendar = new CalendarSplitter(intervalsPerDay, profile, timezoneOffset);
        return SelValues.mapToGraphData(fit, predict, (fitGd, predictGd) -> {
            SeasonalTrend st = new SeasonalTrend(calendar, dropFraction);
            st.fit(fitGd);
            return st.predictAdjusted(predictGd, relMinVar, absMinVar);
        });
    }

    private static SelValue adjusted(
        SelValueWithRange fit,
        SelValueWithRange predict,
        int intervalsPerDay,
        DailyProfile profile,
        Duration timezoneOffset,
        double dropFraction)
    {
        CalendarSplitter calendar = new CalendarSplitter(intervalsPerDay, profile, timezoneOffset);
        return SelValues.mapToGraphData(fit, predict, (fitGd, predictGd) -> {
            SeasonalTrend st = new SeasonalTrend(calendar, dropFraction);
            st.fit(fitGd);
            return st.predictAdjusted(predictGd);
        });
    }

    private static SelValue calculate(ArgsList args) {
        SelValueWithRange fit = args.getWithRange(0);
        SelValueWithRange predict = args.getWithRange(1);
        int intervalsPerDay = (int) args.get(2).castToScalar().getValue();
        String profileName = args.get(3).castToString().getValue();
        DailyProfile profile = DailyProfile.byName(profileName)
            .orElseThrow(() -> new EvaluationException(args.getRange(3), "Profile not found with name: " + profileName));
        Duration timezoneOffset = args.get(4).castToDuration().getDuration();
        double dropFraction = args.get(5).castToScalar().getValue();
        if (args.size() > 6) {
            double relMinVar = args.get(6).castToScalar().getValue();
            double absMinVar = args.get(7).castToScalar().getValue();
            return adjusted(fit, predict, intervalsPerDay, profile, timezoneOffset, dropFraction, relMinVar, absMinVar);
        } else {
            return adjusted(fit, predict, intervalsPerDay, profile, timezoneOffset, dropFraction);
        }
    }

    private static SelFunc function(SelType... args) {
        return SelFunc.newBuilder()
            .name("seasonal_adjusted")
            .help("Return adjusted (that is (x - mean) / std) seasonal data " +
                "with given profile specified by intervals per day count " +
                "and day bucketing scheme")
            .category(SelFuncCategory.PREDICT)
            .args(args)
            .returnType(args[0])
            .handler(SelFnSeasonalAdjusted::calculate)
            .build();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(function(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE));
        registry.add(function(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE));
        registry.add(function(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE));
        registry.add(function(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE));

        registry.add(function(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE));
        registry.add(function(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE));
        registry.add(function(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE));
        registry.add(function(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE, SelTypes.DOUBLE, SelTypes.DOUBLE));
    }
}
