package ru.yandex.solomon.expression.expr.func.analytical;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.value.SelValues;
import ru.yandex.solomon.math.stat.CalendarSplitter;
import ru.yandex.solomon.math.stat.DailyProfile;
import ru.yandex.solomon.math.stat.SeasonalTrend;

/**
 * Return trend of seasonal data with given profile
 * specified by intervals per day count and day bucketing scheme
 * Example usage:
 * <pre>{@code
 *   let source = myMetric{host=cluster};
 *   let fitData = drop_tail(source, 10m);
 *   let trend = seasonal_mean(fitData, source, 12, 'daily', +3h, 0.1);
 * }</pre>
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnSeasonalMean implements SelFuncProvider {
    private static SelValue mean(
            SelValueWithRange fit,
            SelValueWithRange predict,
            int intervalsPerDay,
            DailyProfile profile,
            Duration timezoneOffset,
            double dropFraction)
    {
        CalendarSplitter calendar = new CalendarSplitter(intervalsPerDay, profile, timezoneOffset);
        return SelValues.mapToGraphData(fit, predict, (fitGd, predictGd) -> {
            SeasonalTrend st = new SeasonalTrend(calendar, dropFraction);
            st.fit(fitGd);
            return st.predictMean(predictGd.getTimestamps());
        });
    }

    private static SelValue calculate(ArgsList args) {
        SelValueWithRange fit = args.getWithRange(0);
        SelValueWithRange predict = args.getWithRange(1);
        int intervalsPerDay = (int) args.get(2).castToScalar().getValue();
        String profileName = args.get(3).castToString().getValue();
        DailyProfile profile = DailyProfile.byName(profileName)
            .orElseThrow(() -> new EvaluationException(args.getRange(3), "Profile not found with name: " + profileName));

        Duration timezoneOffset = args.get(4).castToDuration().getDuration();
        double dropFraction = args.get(5).castToScalar().getValue();
        return mean(fit, predict, intervalsPerDay, profile, timezoneOffset, dropFraction);
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("seasonal_mean")
            .help("Return trend of seasonal data with given profile specified " +
                "by intervals per day count and day bucketing scheme")
            .category(SelFuncCategory.PREDICT)
            .args(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE)
            .returnType(SelTypes.GRAPH_DATA)
            .handler(SelFnSeasonalMean::calculate)
            .build());

        registry.add(SelFunc.newBuilder()
            .name("seasonal_mean")
            .help("Return trend of seasonal data with given profile specified " +
                "by intervals per day count and day bucketing scheme")
            .category(SelFuncCategory.PREDICT)
            .args(SelTypes.GRAPH_DATA, SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE)
            .returnType(SelTypes.GRAPH_DATA)
            .handler(SelFnSeasonalMean::calculate)
            .build());

        registry.add(SelFunc.newBuilder()
            .name("seasonal_mean")
            .help("Return trend of seasonal data with given profile specified " +
                "by intervals per day count and day bucketing scheme")
            .category(SelFuncCategory.PREDICT)
            .args(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(SelFnSeasonalMean::calculate)
            .build());

        registry.add(SelFunc.newBuilder()
            .name("seasonal_mean")
            .help("Return trend of seasonal data with given profile specified " +
                "by intervals per day count and day bucketing scheme")
            .category(SelFuncCategory.PREDICT)
            .args(SelTypes.GRAPH_DATA_VECTOR, SelTypes.GRAPH_DATA_VECTOR, SelTypes.DOUBLE, SelTypes.STRING, SelTypes.DURATION, SelTypes.DOUBLE)
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .handler(SelFnSeasonalMean::calculate)
            .build());
    }
}
