package ru.yandex.solomon.expression.expr.func.analytical;

import java.time.Duration;
import java.time.Instant;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.exceptions.PreparingException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueDuration;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.util.time.Interval;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelFnShift implements SelFuncProvider {

    private static NamedGraphData shift(NamedGraphData namedGraphData, Duration window) {
        var source = namedGraphData.getAggrGraphDataArrayList();
        var result = new AggrGraphDataArrayList(source.columnSetMask(), source.length());
        var it = source.iterator();
        var point = RecyclableAggrPoint.newInstance();
        var shift = window.toMillis();
        while (it.next(point)) {
            point.tsMillis += shift;
            result.addRecord(point);
        }
        point.recycle();

        return namedGraphData.toBuilder()
            .setGraphData(result)
            .build();
    }

    private static Duration getShiftWindow(ArgsList params) {
        return Optional.ofNullable(params.get(1))
            .map(SelValue::castToDuration)
            .map(SelValueDuration::getDuration)
            .orElseThrow(() -> new PreparingException(params.getRange(1),
                    "Specified not valid interval for shift function: " + params.get(1)));
    }

    public static Interval interval(Interval interval, ArgsList params) {
        Duration windowSize = getShiftWindow(params);
        Instant start = interval.getBegin().minus(windowSize);
        Instant end = interval.getEnd().minus(windowSize);
        return new Interval(start, end);
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("shift")
            .help("Shift past data on specified time window")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA),
                arg("window").type(SelTypes.DURATION).help("to shift, should be positive for past, and negative for future"))
            .returnType(SelTypes.GRAPH_DATA)
            .interval(SelFnShift::interval)
            .handler(args -> {
                var source = args.get(0).castToGraphData().getNamedGraphData();
                var window = args.get(1).castToDuration().getDuration();
                return new SelValueGraphData(shift(source, window));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("shift")
            .help("Shift past data on specified time window")
            .category(SelFuncCategory.TRANSFORMATION)
            .args(
                arg("source").type(SelTypes.GRAPH_DATA_VECTOR),
                arg("window").type(SelTypes.DURATION).help("to shift, should be positive for past, and negative for future"))
            .returnType(SelTypes.GRAPH_DATA_VECTOR)
            .interval(SelFnShift::interval)
            .handler(args -> {
                var source = args.get(0).castToVector().valueArray();
                var window = args.get(1).castToDuration().getDuration();

                var result = new SelValue[source.length];
                for (int index = 0; index < source.length; index++) {
                    var item = source[index].castToGraphData().getNamedGraphData();
                    result[index] = new SelValueGraphData(shift(item, window));
                }

                return new SelValueVector(SelTypes.GRAPH_DATA, result);
            })
            .build());
    }
}
