package ru.yandex.solomon.expression.expr.func.analytical.histogram;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.EvalContext;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncArgument;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueString;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.math.stat.HistogramBucketCursor;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class HistogramFunc {
    private String name;
    private String help;
    private List<SelFuncArgument> prefixArgs;
    private SelType returnType;
    private Handler handler;
    private Predicate<SelVersion> supportedVersions = SelVersion.MIN::since;

    public enum HistKind {
        LEGACY,
        NATIVE,
        GUESS,
        ;
    }

    public static HistogramFunc newBuilder() {
        return new HistogramFunc();
    }

    static String getBucketKey(SelValueWithRange selValue) {
        String label = ((SelValueString) selValue.getValue()).getValue();
        if (!label.isEmpty() && StringUtils.isBlank(label)) {
            throw new EvaluationException(selValue.getRange(), "label must not be blank");
        }
        return label;
    }

    static List<NamedGraphData> getSource(SelValue selValue) {
        return Arrays.stream(selValue.castToVector().valueArray())
                .map(SelValue::castToGraphData)
                .map(SelValueGraphData::getNamedGraphData)
                .collect(Collectors.toList());
    }

    public HistogramFunc name(String name) {
        this.name = name;
        return this;
    }

    public HistogramFunc help(String help) {
        this.help = help;
        return this;
    }

    public HistogramFunc prefixArgs(SelFuncArgument.Builder... prefixArgs) {
        this.prefixArgs = Stream.of(prefixArgs)
                .map(SelFuncArgument.Builder::build)
                .collect(Collectors.toUnmodifiableList());
        return this;
    }

    @FunctionalInterface
    public interface Handler {
        SelValue apply(EvalContext ctx, ArgsList commonArgs, HistogramArgument histArgs);
    }

    public HistogramFunc handler(Handler handler) {
        this.handler = handler;
        return this;
    }

    public HistogramFunc returnType(SelType type) {
        this.returnType = type;
        return this;
    }

    public HistogramFunc supportedVersions(Predicate<SelVersion> supportedVersions) {
        this.supportedVersions = supportedVersions;
        return this;
    }

    public void provide(SelFuncRegistry registry) {
        SelFuncArgument bucketLabel = SelFuncArgument.arg("bucketLabel")
                .type(SelTypes.STRING)
                .help("Label that identifies histogram buckets")
                .build();

        SelFuncArgument lines = SelFuncArgument.arg("lines")
                .type(SelTypes.GRAPH_DATA_VECTOR)
                .help("Histogram as vector of lines")
                .build();

        SelFuncArgument line = SelFuncArgument.arg("line")
                .type(SelTypes.GRAPH_DATA)
                .help("Histogram as a single line with HISTOGRAM points")
                .build();

        registry.add(SelFunc.newBuilder()
                .name(name)
                .help(help)
                .args(Stream.concat(
                        prefixArgs.stream(),
                        Stream.of(bucketLabel, lines)).toArray(SelFuncArgument[]::new))
                .returnType(returnType)
                .category(SelFuncCategory.COMBINE)
                .supportedVersions(supportedVersions)
                .handler((ctx, args) -> {
                    int argc = args.size();
                    SelValueWithRange bucketKeyArg = args.getWithRange(argc - 2);
                    SelValueWithRange linesArg = args.getWithRange(argc - 1);
                    String bucketKey = getBucketKey(bucketKeyArg);
                    HistogramArgument hist = new HistogramArgument(bucketKey.isEmpty() ? HistKind.NATIVE : HistKind.LEGACY,
                            getSource(linesArg.getValue()), bucketKey, linesArg.getRange());
                    return handler.apply(ctx, args.slice(argc - 2), hist);
                })
        );

        registry.add(SelFunc.newBuilder()
                .name(name)
                .help(help)
                .args(Stream.concat(
                        prefixArgs.stream(),
                        Stream.of(lines)).toArray(SelFuncArgument[]::new))
                .returnType(returnType)
                .category(SelFuncCategory.COMBINE)
                .supportedVersions(supportedVersions)
                .handler((ctx, args) -> {
                    int argc = args.size();
                    SelValueWithRange linesArg = args.getWithRange(argc - 1);
                    HistogramArgument hist = new HistogramArgument(HistKind.GUESS,
                            getSource(linesArg.getValue()), "", linesArg.getRange());
                    return handler.apply(ctx, args.slice(argc - 1), hist);
                })
        );

        registry.add(SelFunc.newBuilder()
                .name(name)
                .help(help)
                .args(Stream.concat(
                        prefixArgs.stream(),
                        Stream.of(line)).toArray(SelFuncArgument[]::new))
                .returnType(returnType)
                .category(SelFuncCategory.COMBINE)
                .supportedVersions(supportedVersions)
                .handler((ctx, args) -> {
                    int argc = args.size();
                    SelValueWithRange lineArg = args.getWithRange(argc - 1);
                    NamedGraphData lineData = lineArg.getValue().castToGraphData().getNamedGraphData();
                    HistogramArgument hist = new HistogramArgument(HistKind.NATIVE, List.of(lineData),
                            "", lineArg.getRange());
                    return handler.apply(ctx, args.slice(argc - 1), hist);
                })
        );
    }

    /**
     * Represents different arguments passed to histogram_* functions
     */
    static class HistogramArgument {
        final HistKind kind;
        final List<NamedGraphData> source;
        final String bucketLabelKeyOrEmpty;
        final PositionRange linesRange;

        HistogramArgument(
                HistKind kind,
                List<NamedGraphData> source,
                String bucketLabelKeyOrEmpty,
                PositionRange linesRange)
        {
            this.kind = kind;
            this.source = source;
            this.bucketLabelKeyOrEmpty = bucketLabelKeyOrEmpty;
            this.linesRange = linesRange;
        }

        boolean isLegacy() {
            return switch (kind) {
                case LEGACY -> true;
                case NATIVE -> false;
                case GUESS -> LegacyHistogramSupport.isLegacyHistograms(source);
            };
        }

        Map<Labels, /* @NotEmpty */ List<HistogramBucketCursor>> getLegacyHistogramBuckets(boolean groupHistograms) {
            return LegacyHistogramSupport.getLegacyHistogramBuckets(groupHistograms, linesRange, source, bucketLabelKeyOrEmpty);
        }
    }
}
