package ru.yandex.solomon.expression.expr.func.analytical.histogram;

import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.func.LabelsUtil;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.math.stat.HistogramBucketCursor;
import ru.yandex.solomon.math.stat.LabelValueNumberPattern;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;

import static java.util.stream.Collectors.filtering;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
class LegacyHistogramSupport {
    private static final EnumSet<MetricType> CUSTOM_HISTOGRAMS = EnumSet.of(
            MetricType.HIST,
            MetricType.HIST_RATE,
            MetricType.LOG_HISTOGRAM
    );

    static boolean isLegacyHistograms(List<NamedGraphData> source) {
        return source.stream()
            .map(LegacyHistogramSupport::metricType)
            .noneMatch(CUSTOM_HISTOGRAMS::contains);
    }

    private static MetricType metricType(NamedGraphData graphData) {
        var dataType = graphData.getDataType();
        if (dataType != MetricType.METRIC_TYPE_UNSPECIFIED) {
            return dataType;
        }

        var metricType = graphData.getType();
        if (metricType != ru.yandex.monlib.metrics.MetricType.UNKNOWN) {
            return MetricTypeConverter.toProto(metricType);
        }

        if (graphData.getAggrGraphDataArrayList().isEmpty()) {
            return MetricType.DGAUGE;
        }

        return MetricType.METRIC_TYPE_UNSPECIFIED;
    }

    private static String computeBucketLabelFromSource(@NotEmpty List<NamedGraphData> source) {
        String differentLabel = "";

        Labels firstLabels = source.get(0).getLabels();
        if (firstLabels.findByKey(LabelKeys.BIN) != null) {
            return LabelKeys.BIN;
        }

        for (int i = 1; i < source.size(); ++i) {
            Labels labels = source.get(i).getLabels();
            for (Label label : labels.toList()) {
                if (LabelKeys.BIN.equals(label.getKey())) {
                    return LabelKeys.BIN;
                }
                if (!label.equals(firstLabels.findBySameKey(label))) {
                    if (differentLabel.isEmpty()) {
                        differentLabel = label.getKey();
                    } else if (!differentLabel.equals(label.getKey())) {
                        return "";
                    }
                }
            }
        }

        return differentLabel;
    }

    static String computeBucketLabelKeyFromLinesAndParam(@NotEmpty List<NamedGraphData> lines, String bucketLabelKey) {
        if (bucketLabelKey.isEmpty()) {
            return computeBucketLabelFromSource(lines);
        }

        return bucketLabelKey;
    }

    @Nullable
    private static HistogramBucketCursor getHistogramBucketFromLineOrNull(String bucketLabelKey, NamedGraphData ngd) {
        Label curBucketLabel = ngd.getLabels().findByKey(bucketLabelKey);
        if (curBucketLabel == null) {
            return null;
        }
        String curBucketLabelValue = curBucketLabel.getValue();
        double bucketLimit;
        try {
            bucketLimit = LabelValueNumberPattern.parse(curBucketLabelValue);
        } catch (NumberFormatException e) {
            return null;
        }
        return new HistogramBucketCursor(bucketLimit, ngd.getDataType(), ngd.getAggrGraphDataArrayList());
    }

    @NotEmpty
    static Map<Labels, List<HistogramBucketCursor>> getLegacyHistogramBuckets(
            boolean groupHistograms,
            PositionRange linesRange,
            @NotEmpty List<NamedGraphData> lines,
            String bucketLabelKeyOrEmpty)
    {
        final String finalBucketLabelKey = computeBucketLabelKeyFromLinesAndParam(lines, bucketLabelKeyOrEmpty);

        if (finalBucketLabelKey.isEmpty()) {
            throw new EvaluationException(linesRange, "It's impossible to calculate histogram function"
                    + " because metrics don't have any valid bucket label:\n"
                    + lines.stream().limit(3).map(NamedGraphData::toString).collect(Collectors.joining("\n")));
        }

        if (groupHistograms) {
            var result = lines.stream()
                .map(ngd -> getHistogramBucketFromLineOrNull(finalBucketLabelKey, ngd))
                .filter(Objects::nonNull)
                .sorted(Comparator.comparing(HistogramBucketCursor::getBucketLimit))
                .collect(toList());
            if (result.isEmpty()) {
                throw new RuntimeException("It's impossible to calculate histogram function because metrics don't have valid bucket label '"
                        + finalBucketLabelKey
                        + "' or don't have it at all");
            }

            Labels commonLabels = LabelsUtil.getCommonLabels(lines);
            return Map.of(commonLabels, result);
        } else {
            return lines.stream()
                    .collect(groupingBy(
                            ngd -> labelsWithoutKey(ngd, finalBucketLabelKey),
                            mapping(
                                    ngd -> getHistogramBucketFromLineOrNull(finalBucketLabelKey, ngd),
                                    filtering(Objects::nonNull, toList()))
                    ))
                    .entrySet().stream()
                    .filter(e -> !e.getValue().isEmpty())
                    .collect(toMap(Map.Entry::getKey, e -> e.getValue().stream()
                            .sorted(Comparator.comparing(HistogramBucketCursor::getBucketLimit))
                            .collect(toList())));
        }
    }

    private static Labels labelsWithoutKey(NamedGraphData ngd, String labelKey) {
        return ngd.getLabels().removeByKey(labelKey);
    }

}
