package ru.yandex.solomon.expression.expr.func.analytical.histogram;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.func.SelFuncArgument;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnHistogramCumulativeDistribution implements SelFuncProvider {

    private static final String CDFP = "histogram_cdfp";
    private static final String COUNT = "histogram_count";

    @Override
    public void provide(SelFuncRegistry registry) {
        SelFuncArgument.Builder from = SelFuncArgument
                .arg("from")
                .type(SelTypes.DOUBLE)
                .help("Lower bound");

        SelFuncArgument.Builder to = SelFuncArgument
                .arg("to")
                .type(SelTypes.DOUBLE)
                .help("Upper bound");

        SelFuncArgument.Builder fromVec = SelFuncArgument
                .arg("from")
                .type(SelTypes.DOUBLE_VECTOR)
                .help("Lower bounds");

        SelFuncArgument.Builder toVec = SelFuncArgument
                .arg("to")
                .type(SelTypes.DOUBLE_VECTOR)
                .help("Upper bounds");

        for (var arg1 : new SelFuncArgument.Builder[] {from, fromVec}) {
            for (var arg2 : new SelFuncArgument.Builder[] {to, toVec}) {
                HistogramFunc.newBuilder()
                        .name(CDFP)
                        .help("Compute the percentage of histogram values between bounds")
                        .prefixArgs(arg1, arg2)
                        .returnType(SelTypes.GRAPH_DATA_VECTOR)
                        .handler((ctx, args, hist) -> {
                            var bounds = broadcast(args.getWithRange(0), args.getWithRange(1));
                            return handler(ctx.getVersion(), PositionRange.convexHull(args.getRange(0), args.getRange(1)),
                                    hist, bounds.getLeft(), bounds.getRight(), true);
                        })
                        .provide(registry);
                HistogramFunc.newBuilder()
                        .name(COUNT)
                        .help("Compute the count of histogram values between bounds")
                        .prefixArgs(arg1, arg2)
                        .returnType(SelTypes.GRAPH_DATA_VECTOR)
                        .handler((ctx, args, hist) -> {
                            var bounds = broadcast(args.getWithRange(0), args.getWithRange(1));
                            return handler(ctx.getVersion(), PositionRange.convexHull(args.getRange(0), args.getRange(1)),
                                    hist, bounds.getLeft(), bounds.getRight(), false);
                        })
                        .provide(registry);
            }
        }

        for (var arg : new SelFuncArgument.Builder[] {to, toVec}) {
            HistogramFunc.newBuilder()
                    .name(CDFP)
                    .help("Compute the percentage of histogram values below bound")
                    .prefixArgs(arg)
                    .returnType(SelTypes.GRAPH_DATA_VECTOR)
                    .handler((ctx, args, hist) -> {
                        var bounds = broadcast(args.getWithRange(0));
                        return handler(ctx.getVersion(), args.getRange(0), hist, bounds.getLeft(), bounds.getRight(), true);
                    })
                    .provide(registry);
            HistogramFunc.newBuilder()
                    .name(COUNT)
                    .help("Compute the count of values in histogram")
                    .prefixArgs(arg)
                    .returnType(SelTypes.GRAPH_DATA_VECTOR)
                    .handler((ctx, args, hist) -> {
                        var bounds = broadcast(args.getWithRange(0));
                        return handler(ctx.getVersion(), args.getRange(0), hist, bounds.getLeft(), bounds.getRight(), false);
                    })
                    .provide(registry);
        }

        HistogramFunc.newBuilder()
                .name(COUNT)
                .help("Compute the percentage of the histogram values below bound")
                .prefixArgs()
                .returnType(SelTypes.GRAPH_DATA_VECTOR)
                .handler((ctx, args, hist) -> {
                    double[] lowerBounds = new double[] {Double.NEGATIVE_INFINITY};
                    double[] upperBounds = new double[] {Double.POSITIVE_INFINITY};
                    return handler(ctx.getVersion(), PositionRange.UNKNOWN, hist, lowerBounds, upperBounds, false);
                })
                .provide(registry);
    }

    private SelValueVector handler(
            SelVersion version,
            PositionRange boundsRange,
            HistogramFunc.HistogramArgument hist,
            double[] lowerBounds,
            double[] upperBounds,
            boolean normalize)
    {
        validateBounds(boundsRange, lowerBounds, upperBounds);
        boolean grouping = SelVersion.HISTOGRAM_FUNCTIONS_DONT_MERGE_3.before(version);
        var factory = HistogramCumulativeDistributionIteratorFactory.of(normalize ? CDFP : COUNT,
                grouping, lowerBounds, upperBounds, normalize);
        var iterators = factory.makeIteratorForDataType(hist);

        List<SelValueGraphData> result = new ArrayList<>(lowerBounds.length * iterators.size());
        iterators.forEach((labels, iterator) -> {
            List<AggrGraphDataArrayList> resultData = IteratorHelpers.consumeToListAggrGraphData(iterator,
                    lowerBounds.length);

            for (int i = 0; i < lowerBounds.length; i++) {
                var gd = resultData.get(i);
                NamedGraphData.Builder ngd = NamedGraphData.newBuilder()
                        .setType(ru.yandex.monlib.metrics.MetricType.DGAUGE)
                        .setGraphData(gd.isEmpty() ? MetricType.METRIC_TYPE_UNSPECIFIED : MetricType.DGAUGE, gd);

                if (grouping) {
                    String alias = String.format("from %s to %s", lowerBounds[i], upperBounds[i]);
                    ngd.setLabels(labels);
                    ngd.setAlias(alias);
                } else {
                    LabelsBuilder builder = labels.toBuilder();
                    if (Double.isFinite(lowerBounds[i])) {
                        builder.add("lower_bound", formatBound(lowerBounds[i]));
                    }
                    if (Double.isFinite(upperBounds[i])) {
                        builder.add("upper_bound", formatBound(upperBounds[i]));
                    }
                    ngd.setLabels(builder.build());
                }

                result.add(new SelValueGraphData(ngd.build()));
            }
        });

        return new SelValueVector(SelTypes.GRAPH_DATA, result.toArray(SelValue[]::new));
    }

    private static String formatBound(double value) {
        if (value >= 0) {
            return String.format("%s", value);
        }
        // XXX: needs better solution
        return String.format("_%s", -value);
    }

    private void validateBounds(PositionRange boundsRange, double[] left, double[] right) {
        if (left.length != right.length) {
            throw new EvaluationException(boundsRange, "Bounds array differ in length");
        }
        for (int i = 0; i < left.length; i++) {
            if (Double.isNaN(left[i])) {
                throw new EvaluationException(boundsRange, "invalid lower bound: " + left[i]);
            }
            if (Double.isNaN(right[i])) {
                throw new EvaluationException(boundsRange, "invalid upper bound: " + left[i]);
            }
            if (Double.compare(left[i], right[i]) > 0) {
                throw new EvaluationException(boundsRange, "the lower bound " + left[i] +
                        " is greater than the upper one " + right[i]);
            }
        }
    }

    private double[] toVector(SelValueWithRange valueWithRange) {
        SelValue value = valueWithRange.getValue();
        if (value.type() == SelTypes.DOUBLE) {
            return new double[] {value.castToScalar().getValue()};
        }
        double[] vector = value.castToVector().doubleArray();
        if (vector.length == 0) {
            throw new EvaluationException(valueWithRange.getRange(), "Bounds cannot be empty");
        }
        return vector;
    }

    private Pair<double[],double[]> broadcast(SelValueWithRange from, SelValueWithRange to) {
        double[] froms = toVector(from);
        double[] tos = toVector(to);
        if (froms.length == tos.length) {
            return Pair.of(froms, tos);
        }
        if (froms.length != 1 && tos.length != 1) {
            throw new EvaluationException(PositionRange.convexHull(from.getRange(), to.getRange()),
                    "Bounds arrays have different lengths");
        }
        if (froms.length == 1) {
            return Pair.of(Arrays.stream(tos).map(ignore -> froms[0]).toArray(), tos);
        } else { // tos.length == 1
            return Pair.of(froms, Arrays.stream(froms).map(ignore -> tos[0]).toArray());
        }
    }

    private Pair<double[],double[]> broadcast(SelValueWithRange to) {
        double[] tos = toVector(to);
        double[] froms = Arrays.stream(tos).map(ignore -> Double.NEGATIVE_INFINITY).toArray();
        return Pair.of(froms, tos);
    }
}
