package ru.yandex.solomon.expression.expr.func.analytical.histogram;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.func.SelFuncArgument;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.expression.value.SelValueWithRange;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.iterator.GenericIterator;

/**
 * <p>Apply percentile as aggregate function for values at the same time.
 * <p>
 * <p>Example usage {@code histogram_percentile(99.9, 'bin', group_by_time(15s, 'sum', graphDataVector))}
 * where 'bin' it's label that as a value contains right border for bucket
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SelFnHistogramPercentile implements SelFuncProvider {

    private static final String NAME = "histogram_percentile";

    private List<NamedGraphData> handler(
            SelVersion version,
            double[] percentileLevels,
            HistogramFunc.HistogramArgument hist)
    {
        boolean grouping = SelVersion.HISTOGRAM_FUNCTIONS_DONT_MERGE_3.before(version);
        var factory = new HistogramPercentileIteratorFactory(NAME, grouping, percentileLevels);
        var percentilesIterator = factory.makeIteratorForDataType(hist);
        List<NamedGraphData> result = new ArrayList<>(percentileLevels.length * percentilesIterator.size());

        if (grouping && percentilesIterator.isEmpty()) { // Legacy behavior for empty input
            percentilesIterator = Map.of(Labels.empty(), GenericIterator.empty());
        }

        percentilesIterator.forEach((labels, iterator) -> {
            List<AggrGraphDataArrayList> percentilesData = IteratorHelpers.consumeToListAggrGraphData(iterator, percentileLevels.length);

            for (int i = 0; i < percentilesData.size(); i++) {
                var gd = percentilesData.get(i);

                NamedGraphData.Builder ngd = NamedGraphData.newBuilder()
                        .setType(ru.yandex.monlib.metrics.MetricType.DGAUGE)
                        .setGraphData(gd.isEmpty() ? MetricType.METRIC_TYPE_UNSPECIFIED : MetricType.DGAUGE, gd);

                String percentile = String.format("%s", percentileLevels[i]);
                if (grouping) {
                    ngd.setLabels(labels);
                    ngd.setAlias("p" + percentile);
                } else {
                    ngd.setLabels(labels.add("percentile", percentile));
                }

                result.add(ngd.build());
            }
        });

        return result;
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        HistogramFunc.newBuilder()
                .name(NAME)
                .help("Apply percentile as aggregate function for values at the same time")
                .prefixArgs(SelFuncArgument
                        .arg("percentileLevel")
                        .type(SelTypes.DOUBLE)
                        .help("Level of the percentile to compute, should be between 0 and 100"))
                .supportedVersions(SelVersion.GROUP_LINES_RETURN_VECTOR_2::before)
                .returnType(SelTypes.GRAPH_DATA)
                .handler((ctx, args, hist) -> {
                    double[] percentiles = {getPercentileLevel(args.getWithRange(0))};
                    var result = handler(ctx.getVersion(), percentiles, hist);
                    return new SelValueGraphData(result.get(0));
                })
                .provide(registry);

        HistogramFunc.newBuilder()
                .name(NAME)
                .help("Apply percentile as aggregate function for values at the same time")
                .prefixArgs(SelFuncArgument
                        .arg("percentileLevel")
                        .type(SelTypes.DOUBLE)
                        .help("Level of the percentile to compute, should be between 0 and 100"))
                .supportedVersions(SelVersion.GROUP_LINES_RETURN_VECTOR_2::since)
                .returnType(SelTypes.GRAPH_DATA_VECTOR)
                .handler((ctx, args, hist) -> {
                    double[] percentiles = {getPercentileLevel(args.getWithRange(0))};
                    var result = handler(ctx.getVersion(), percentiles, hist);
                    return new SelValueVector(SelTypes.GRAPH_DATA, result.stream()
                            .map(SelValueGraphData::new)
                            .toArray(SelValueGraphData[]::new));
                })
                .provide(registry);

        HistogramFunc.newBuilder()
                .name(NAME)
                .help("Apply percentile as aggregate function for values at the same time")
                .prefixArgs(SelFuncArgument
                        .arg("percentileLevels")
                        .type(SelTypes.DOUBLE_VECTOR)
                        .help("Levels of the percentiles to compute, all should be between 0 and 100 and sorted"))
                .returnType(SelTypes.GRAPH_DATA_VECTOR)
                .handler((ctx, args, hist) -> {
                    double[] percentiles = getPercentileLevels(args.getWithRange(0));
                    var result = handler(ctx.getVersion(), percentiles, hist);
                    return new SelValueVector(SelTypes.GRAPH_DATA, result.stream()
                            .map(SelValueGraphData::new)
                            .toArray(SelValueGraphData[]::new));
                })
                .provide(registry);
    }

    private static void validatePercentileLevel(PositionRange range, double p) {
        if (p < 0) {
            throw new EvaluationException(range, "percentile level must not be negative: " + p);
        }
        if (p > 100) {
            throw new EvaluationException(range, "percentile level must not be greater than 100: " + p);
        }
    }

    public static double getPercentileLevel(SelValueWithRange selValue) {
        double p = selValue.getValue().castToScalar().getValue();
        validatePercentileLevel(selValue.getRange(), p);
        return p;
    }

    private static double[] getPercentileLevels(SelValueWithRange selValue) {
        double[] percentiles = selValue.getValue().castToVector().doubleArray();
        PositionRange range = selValue.getRange();
        for (int i = 0; i < percentiles.length; i++) {
            double percentile = percentiles[i];
            validatePercentileLevel(range, percentile);
            if (i > 0 && Double.compare(percentiles[i - 1], percentiles[i]) >= 0) {
                throw new EvaluationException(range, "Percentile should be sorted: " + Arrays.toString(percentiles));
            }
        }
        return percentiles;
    }
}
