package ru.yandex.solomon.expression.expr.func.analytical.histogram;

import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.math.stat.HistogramAggr;
import ru.yandex.solomon.model.point.MutableDataPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.iterator.GenericIterator;

/**
 * <p>Estimate sum of values stored in histogram at each time point.
 * <p>
 * <p>Example usage {@code histogram_sum('bin', legacyHistogramVector)},
 * where 'bin' is the label key whose value contains right border for bucket
 * {@code histogram_sum(legacyHistogramVector)}, {@code histogram_sum(nativeHistogram)}
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnHistogramSum implements SelFuncProvider {

    private static final String NAME = "histogram_sum";
    private static final HistogramReduceIteratorFactory SUM_ITERATOR_FACTORY_GROUPING =
            HistogramReduceIteratorFactory.of(NAME, true, HistogramAggr::sum, HistogramAggr::sum, HistogramAggr::sum);
    private static final HistogramReduceIteratorFactory SUM_ITERATOR_FACTORY_VECTORED =
            HistogramReduceIteratorFactory.of(NAME, false, HistogramAggr::sum, HistogramAggr::sum, HistogramAggr::sum);

    @Override
    public void provide(SelFuncRegistry registry) {
        HistogramFunc.newBuilder()
                .name(NAME)
                .help("Estimate sum of values stored in histogram at each time point")
                .prefixArgs()
                .returnType(SelTypes.GRAPH_DATA_VECTOR)
                .handler((ctx, emptyArgs, hist) -> handler(ctx.getVersion(), hist))
                .provide(registry);
    }

    private static SelValueVector handler(SelVersion version, HistogramFunc.HistogramArgument hist) {
        var itFactory = SelVersion.HISTOGRAM_FUNCTIONS_DONT_MERGE_3.since(version)
                ? SUM_ITERATOR_FACTORY_VECTORED
                : SUM_ITERATOR_FACTORY_GROUPING;
        Map<Labels, GenericIterator<MutableDataPoint>> iterators = itFactory.makeIteratorForDataType(hist);

        SelValue[] results = iterators.entrySet().stream()
                .map(e -> {
                    Labels labels = e.getKey();
                    GenericIterator<MutableDataPoint> iterator = e.getValue();
                    AggrGraphDataArrayList result = IteratorHelpers.consumeToAggrGraphData(iterator);
                    return NamedGraphData.newBuilder()
                            .setLabels(labels)
                            .setType(ru.yandex.monlib.metrics.MetricType.DGAUGE)
                            .setGraphData(MetricType.DGAUGE, result)
                            .build();
                })
                .map(SelValueGraphData::new)
                .toArray(SelValue[]::new);

        return new SelValueVector(SelTypes.GRAPH_DATA, results);
    }
}
