package ru.yandex.solomon.expression.expr.func.analytical.sideEffect;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValueStatus;

/**
 * Check condition and set alert evaluation status if condition was true.
 * Code evaluation is terminated immediately. Usage:
 * <code>
 * let data = {...};
 * no_data_if(count(data) == 0);
 * let val = avg(data);
 * alarm_if(val > 95);
 * </code>
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnStatusIf implements SelFuncProvider {

    private static SelFunc function(String name, SelValueStatus status) {
        return SelFunc.newBuilder()
            .name(name)
            .category(SelFuncCategory.OTHER)
            .help("Check condition and set alert evaluation status if condition was true")
            .args(SelTypes.BOOLEAN)
            .returnType(SelTypes.SIDE_EFFECT)
            .handler(args -> {
                var condition = args.get(0).castToBoolean();
                return condition.getValue() ? status : SelValueStatus.UNKNOWN;
            })
            .build();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(function("ok_if", SelValueStatus.OK));
        registry.add(function("warn_if", SelValueStatus.WARN));
        registry.add(function("alarm_if", SelValueStatus.ALARM));
        registry.add(function("no_data_if", SelValueStatus.NO_DATA));
    }
}
