package ru.yandex.solomon.expression.expr.func.analytical.summary;

import java.util.Arrays;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnSummaryExtract implements SelFuncProvider {
    private static final Logger logger = LoggerFactory.getLogger(SelFnSummaryExtract.class);

    enum SummaryComponent {
        LAST, MIN, MAX, SUM, COUNT, AVG;
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        for (var component : SummaryComponent.values()) {
            provideSingle(registry, component.name().toLowerCase(), ngd -> {
                if (ngd.getAggrGraphDataArrayList().isEmpty()) {
                    return ngd.toBuilder()
                            .setGraphData(MetricType.DGAUGE, AggrGraphDataArrayList.empty())
                            .build();
                }
                switch (ngd.getDataType()) {
                    case DSUMMARY:
                        return SummaryExtractors.extractDoubleSummary(component, ngd);
                    case ISUMMARY:
                        return SummaryExtractors.extractInt64Summary(component, ngd);
                    default:
                        throw new IllegalArgumentException("Cannot extract component " + component + " from " + ngd.getDataType());
                }
            });
        }
    }

    private void provideSingle(SelFuncRegistry registry, String func, Function<NamedGraphData, NamedGraphData> extractor) {
        registry.add(SelFunc.newBuilder()
                .name("summary_" + func)
                .help("Extract SUMMARY:" + func + " component")
                .category(SelFuncCategory.OTHER)
                .args(SelTypes.GRAPH_DATA_VECTOR)
                .returnType(SelTypes.GRAPH_DATA_VECTOR)
                .pure(true)
                .handler(args -> {
                    try {
                        return new SelValueVector(SelTypes.GRAPH_DATA,
                                Arrays.stream(args.get(0).castToVector().valueArray())
                                        .map(selValue -> selValue.castToGraphData().getNamedGraphData())
                                        .map(extractor)
                                        .map(SelValueGraphData::new)
                                        .toArray(SelValue[]::new));
                    } catch (Exception e) {
                        logger.error("Extracting from summary failed", e);
                        throw new EvaluationException(args.getRange(0), e);
                    }
                })
                .build());

        registry.add(SelFunc.newBuilder()
                .name("summary_" + func)
                .help("Extract SUMMARY:" + func + " component")
                .category(SelFuncCategory.OTHER)
                .args(SelTypes.GRAPH_DATA)
                .returnType(SelTypes.GRAPH_DATA)
                .pure(true)
                .handler(args -> {
                    try{
                        return new SelValueGraphData(extractor.apply(args.get(0).castToGraphData().getNamedGraphData()));
                    } catch (Exception e) {
                        logger.error("Extracting from summary failed", e);
                        throw new EvaluationException(args.getRange(0), e);
                    }
                })
                .build());
    }
}
