package ru.yandex.solomon.expression.expr.func.analytical.summary;

import java.util.EnumMap;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.ToDoubleFunction;
import java.util.function.ToLongFunction;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.expr.func.analytical.summary.SelFnSummaryExtract.SummaryComponent;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
class SummaryExtractors {

    private static final EnumMap<SummaryComponent, Pair<MetricType, Consumer<AggrPoint>>> DOUBLE_SUMMARY_EXTRACTORS =
            new EnumMap<>(Map.of(
                    SummaryComponent.LAST, ofDoubleSummary(SummaryDoubleSnapshot::getLast, Double.NaN),
                    SummaryComponent.MIN, ofDoubleSummary(SummaryDoubleSnapshot::getMin, Double.POSITIVE_INFINITY),
                    SummaryComponent.MAX, ofDoubleSummary(SummaryDoubleSnapshot::getMax, Double.NEGATIVE_INFINITY),
                    SummaryComponent.SUM, ofDoubleSummary(SummaryDoubleSnapshot::getSum, 0d),
                    SummaryComponent.AVG, ofDoubleSummary(s -> s.getSum() / s.getCount(), Double.NaN),
                    SummaryComponent.COUNT, ofDoubleSummary(SummaryDoubleSnapshot::getCount, 0)
            ));

    private static final EnumMap<SummaryComponent, Pair<MetricType, Consumer<AggrPoint>>> INT64_SUMMARY_EXTRACTORS =
            new EnumMap<>(Map.of(
                    SummaryComponent.LAST, ofInt64Summary(SummaryInt64Snapshot::getLast, Double.NaN),
                    SummaryComponent.MIN, ofInt64Summary(SummaryInt64Snapshot::getMin, Long.MAX_VALUE),
                    SummaryComponent.MAX, ofInt64Summary(SummaryInt64Snapshot::getMax, Long.MIN_VALUE),
                    SummaryComponent.SUM, ofInt64Summary(SummaryInt64Snapshot::getSum, 0),
                    SummaryComponent.AVG, ofInt64Summary(s -> ((double)s.getSum()) / s.getCount(), Double.NaN),
                    SummaryComponent.COUNT, ofInt64Summary(SummaryInt64Snapshot::getCount, 0)
            ));

    static NamedGraphData extractDoubleSummary(SummaryComponent component, NamedGraphData ngd) {
        Pair<MetricType, Consumer<AggrPoint>> extractor = DOUBLE_SUMMARY_EXTRACTORS.get(component);
        if (extractor != null) {
            return extractToType(ngd, extractor.getLeft(), extractor.getRight());
        }
        throw new IllegalArgumentException("Unsupported component " + component + " for DSUMMARY");
    }

    static NamedGraphData extractInt64Summary(SummaryComponent component, NamedGraphData ngd) {
        Pair<MetricType, Consumer<AggrPoint>> extractor = INT64_SUMMARY_EXTRACTORS.get(component);
        if (extractor != null) {
            return extractToType(ngd, extractor.getLeft(), extractor.getRight());
        }
        throw new IllegalArgumentException("Unsupported component " + component + " for ISUMMARY");
    }

    private static NamedGraphData extractToType(NamedGraphData ngd, MetricType targetType, Consumer<AggrPoint> extractor) {
        var targetTypeProto = MetricTypeConverter.toNotNullProto(targetType);
        int targetMask = StockpileColumns.minColumnSet(targetTypeProto);
        AggrGraphDataArrayListOrView source = ngd.getAggrGraphDataArrayList();
        AggrGraphDataArrayList result = new AggrGraphDataArrayList(targetMask, source.length());

        var iterator = new ExtractIterator(targetMask, source.iterator(), extractor);
        result.addAllFrom(iterator);

        return ngd.toBuilder()
                .setType(targetType)
                .setGraphData(targetTypeProto, result)
                .build();
    }

    private static class ExtractIterator extends AggrGraphDataListIterator {
        private final Consumer<AggrPoint> pointExtractor;
        private final AggrGraphDataListIterator iterator;

        ExtractIterator(
                int columnSetMask,
                AggrGraphDataListIterator iterator,
                Consumer<AggrPoint> pointExtractor)
        {
            super(columnSetMask);
            this.pointExtractor = pointExtractor;
            this.iterator = iterator;
        }

        @Override
        public boolean next(AggrPoint target) {
            if (iterator.next(target)) {
                pointExtractor.accept(target);
                return true;
            }
            return false;
        }
    }

    private static <T> double map(@Nullable T source, ToDoubleFunction<T> mapper, double defValue) {
        if (source == null) {
            return defValue;
        }
        return mapper.applyAsDouble(source);
    }

    private static <T> long map(@Nullable T source, ToLongFunction<T> mapper, long defValue) {
        if (source == null) {
            return defValue;
        }
        return mapper.applyAsLong(source);
    }

    private static Pair<MetricType, Consumer<AggrPoint>> ofDoubleSummary(ToDoubleFunction<SummaryDoubleSnapshot> extractor, double defValue) {
        return Pair.of(MetricType.DGAUGE, point -> {
            point.setValue(map(point.summaryDouble, extractor, defValue));
            point.clearField(StockpileColumn.DSUMMARY);
        });
    }

    private static Pair<MetricType, Consumer<AggrPoint>> ofDoubleSummary(ToLongFunction<SummaryDoubleSnapshot> extractor, long defValue) {
        return Pair.of(MetricType.IGAUGE, point -> {
            point.setLongValue(map(point.summaryDouble, extractor, defValue));
            point.clearField(StockpileColumn.DSUMMARY);
        });
    }

    private static Pair<MetricType, Consumer<AggrPoint>> ofInt64Summary(ToDoubleFunction<SummaryInt64Snapshot> extractor, double defValue) {
        return Pair.of(MetricType.DGAUGE, point -> {
            point.setValue(map(point.summaryInt64, extractor, defValue));
            point.clearField(StockpileColumn.ISUMMARY);
        });
    }

    private static Pair<MetricType, Consumer<AggrPoint>> ofInt64Summary(ToLongFunction<SummaryInt64Snapshot> extractor, long defValue) {
        return Pair.of(MetricType.IGAUGE, point -> {
            point.setLongValue(map(point.summaryInt64, extractor, defValue));
            point.clearField(StockpileColumn.ISUMMARY);
        });
    }

}
