package ru.yandex.solomon.expression.expr.func.analytical.trends;

import java.time.Duration;
import java.util.Arrays;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncArgument;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.model.timeseries.GraphData;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public abstract class SelFnTrend implements SelFuncProvider {

    protected abstract String name();

    protected abstract String help();

    protected abstract GraphData computeTrendLine(GraphData graphData, Duration fromDuration, Duration toDuration);

    @Override
    public void provide(SelFuncRegistry registry) {
        Supplier<SelFunc.Builder> buildPartial = () -> SelFunc.newBuilder()
                .name(name())
                .help(help())
                .category(SelFuncCategory.DEPRECATED); // TODO: reconsider these functions

        SelFuncArgument source = arg("source").help("Source data to fit").type(SelTypes.GRAPH_DATA).build();
        SelFuncArgument sources = arg("sources").help("Source data to fit").type(SelTypes.GRAPH_DATA_VECTOR).build();
        SelFuncArgument from = arg("predict_begin")
                .help("First predicted point timestamp, relative to the last source point timestamp")
                .type(SelTypes.DURATION)
                .build();
        SelFuncArgument to = arg("predict_end")
                .help("Last predicted point timestamp, relative to the last source point timestamp")
                .type(SelTypes.DURATION)
                .build();

        registry.add(buildPartial.get()
                .args(source, from, to)
                .returnType(SelTypes.GRAPH_DATA)
                .handler(args -> {
                    GraphData graphData = args.get(0).castToGraphData().getGraphData();

                    Duration fromDuration = args.get(1).castToDuration().getDuration();
                    Duration toDuration = args.get(2).castToDuration().getDuration();

                    GraphData trendGraphData = computeTrendLine(graphData, fromDuration, toDuration);

                    return new SelValueGraphData(trendGraphData);
                })
                .build());

        registry.add(buildPartial.get()
                .args(sources, from, to)
                .returnType(SelTypes.GRAPH_DATA_VECTOR)
                .handler(args -> {
                    SelValue[] graphDatas = args.get(0).castToVector().valueArray();

                    Duration fromDuration = args.get(1).castToDuration().getDuration();
                    Duration toDuration = args.get(2).castToDuration().getDuration();

                    SelValue[] trendGraphDatas = Arrays.stream(graphDatas)
                            .map(selValue -> selValue.castToGraphData().getGraphData())
                            .map(graphData -> computeTrendLine(graphData, fromDuration, toDuration))
                            .map(SelValueGraphData::new)
                            .toArray(SelValue[]::new);

                    return new SelValueVector(SelTypes.GRAPH_DATA, trendGraphDatas);
                })
                .build());
    }
}
