package ru.yandex.solomon.expression.expr.func.string;

import java.util.Formatter;
import java.util.Locale;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValueString;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * <p>The to_fixed(num, digits) method formats a number using fixed-point notation.
 *
 * <p>Where <b>digits</b> - the number of digits to appear after the decimal point,
 * this may be a value between 0 and 20, inclusive.
 *
 * <p>Example usage {@code to_fixed(12.13, 1)} that returns 12.1
 *
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class SelFnToFixed implements SelFuncProvider {

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("to_fixed")
            .help("The to_fixed(num, digits) method formats a number using fixed-point notation")
            .category(SelFuncCategory.OTHER)
            .args(
                arg("num").type(SelTypes.DOUBLE).help("number to format"),
                arg("digits").type(SelTypes.DOUBLE)
                    .help("the number of digits to appear after the decimal point,"
                        + "this may be a value between 0 and 20, inclusive."))
            .returnType(SelTypes.STRING)
            .handler(args -> {
                double value = args.get(0).castToScalar().getValue();
                int digits = (int) args.get(1).castToScalar().getValue();

                String formatted = new Formatter(Locale.ROOT).format("%." + digits + "f", value).toString();
                return new SelValueString(formatted);
            })
            .build());
    }
}
