package ru.yandex.solomon.expression.expr.func.util;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.ast.AstCall;
import ru.yandex.solomon.expression.ast.AstIdent;
import ru.yandex.solomon.expression.ast.AstValueDouble;
import ru.yandex.solomon.expression.expr.IntrinsicsExternalizer;
import ru.yandex.solomon.expression.expr.SelExpr;
import ru.yandex.solomon.expression.expr.SelExprParam;
import ru.yandex.solomon.expression.expr.SelExprValue;
import ru.yandex.solomon.expression.expr.SelFunctions;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncArgument;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValueDouble;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.model.timeseries.GraphData;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * <p>Returns current alert evaluation history as a timeseries. Returns empty timeseries
 * if called not from an alert context.
 * <p>Example usage {@code
 * let alert_statuses = alert_evaluation_history();
 * let problems = heaviside(alert_statuses - 1);  // maps NO DATA / OK -> 0, other -> 1
 * let good_points = drop_if(problems, data);  // Drop data when there were known problems (alarms)
 * }
 */
@ParametersAreNonnullByDefault
public class SelFnAlertEvaluationHistory implements SelFuncProvider {

    private static final String NAME = "alert_evaluation_history";
    private static final String HELP = "Returns alert evaluation history as a timeseries. Returns empty timeseries " +
            "if called not from an alert context.";

    public static final int ALERT_STATUS_LOWER_BITS = 6;
    public static final int ALERT_STATUS_DIVISOR = 1 << ALERT_STATUS_LOWER_BITS;

    private static Selectors getInlinedSelectors(ArgsList args, IntrinsicsExternalizer externalizer) {
        if (externalizer.getAlertingStatuses() == null) {
            return null;
        }

        List<Selector> selectors = makeSelectorsFromAlertingStatuses(externalizer.getAlertingStatuses());

        if (args.size() == 0) {
            // Take from evaluation context (refer this)
            String projectId = externalizer.getProjectId();
            String alertId = externalizer.getAlertId();
            String parentId = externalizer.getParentId();

            if (projectId == null || alertId == null || parentId == null) {
                return null;
            }

            selectors.add(Selector.exact("projectId", projectId));
            selectors.add(Selector.exact("alertId", alertId));
            addExactOrAbsent(selectors, "parentId", parentId);
        } else {
            // Take from function arguments (refer other)
            String projectId = args.get(0).castToString().getValue();
            String alertId = args.get(1).castToString().getValue();

            addGlobOrAbsent(selectors, "projectId", projectId);
            addGlobOrAbsent(selectors, "alertId", alertId);
            if (args.size() > 2) {
                String parentId = args.get(2).castToString().getValue();
                addGlobOrAbsent(selectors, "parentId", parentId);
            }
        }

        return Selectors.of(selectors);
    }

    private static SelExpr postProcess(SelVersion version, ArgsList argsList, SelExprParam param) {
        // return mod(param, ALERT_STATUS_MULTIPLIER)
        AstValueDouble divisorAst = new AstValueDouble(param.getRange(), ALERT_STATUS_DIVISOR);
        SelExprValue divisor = new SelExprValue(divisorAst, new SelValueDouble(divisorAst.getValue()));
        AstIdent mod = new AstIdent(param.getRange(), "mod");
        AstCall call = new AstCall(param.getRange(), mod, List.of(param.getSourceAst(), divisorAst));
        return SelFunctions.exprCall(version, call, List.of(param, divisor));
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
                .name(NAME)
                .help(HELP)
                .category(SelFuncCategory.OTHER)
                .pure(false)
                .returnType(SelTypes.GRAPH_DATA)
                .handler(args -> new SelValueGraphData(GraphData.empty))
                .externalizer(SelFnAlertEvaluationHistory::getInlinedSelectors)
                .externalizerPostProcess(SelFnAlertEvaluationHistory::postProcess)
                .build());

        SelFuncArgument projectId = arg("projectId")
                .type(SelTypes.STRING)
                .help("project id, may be glob-selector")
                .build();

        SelFuncArgument alertId = arg("alertId")
                .type(SelTypes.STRING)
                .help("alert id, may be glob-selector")
                .build();

        SelFuncArgument parentId = arg("parentId")
                .type(SelTypes.STRING)
                .help("parent alert id, may be glob-selector")
                .build();

        registry.add(SelFunc.newBuilder()
                .name(NAME)
                .help(HELP)
                .category(SelFuncCategory.OTHER)
                .pure(false)
                .args(projectId, alertId)
                .returnType(SelTypes.GRAPH_DATA_VECTOR)
                .handler(args -> new SelValueGraphData(GraphData.empty))
                .externalizer(SelFnAlertEvaluationHistory::getInlinedSelectors)
                .externalizerPostProcess(SelFnAlertEvaluationHistory::postProcess)
                .build());

        registry.add(SelFunc.newBuilder()
                .name(NAME)
                .help(HELP)
                .category(SelFuncCategory.OTHER)
                .pure(false)
                .args(projectId, alertId, parentId)
                .returnType(SelTypes.GRAPH_DATA_VECTOR)
                .handler(args -> new SelValueGraphData(GraphData.empty))
                .externalizer(SelFnAlertEvaluationHistory::getInlinedSelectors)
                .externalizerPostProcess(SelFnAlertEvaluationHistory::postProcess)
                .build());
    }

    private static List<Selector> makeSelectorsFromAlertingStatuses(ShardKey alertingStatuses) {
        String alertingProject = alertingStatuses.getProject();
        String alertingCluster = alertingStatuses.getCluster();
        String alertingService = alertingStatuses.getService();

        List<Selector> selectors = new ArrayList<>(7);

        selectors.add(Selector.exact(LabelKeys.PROJECT, alertingProject));
        selectors.add(Selector.exact(LabelKeys.CLUSTER, alertingCluster));
        selectors.add(Selector.exact(LabelKeys.SERVICE, alertingService));
        selectors.add(Selector.exact(LabelKeys.SENSOR, "alert.evaluation.status"));

        return selectors;
    }

    private static void addGlobOrAbsent(List<Selector> selectors, String key, String value) {
        if (value.isEmpty()) {
            selectors.add(Selector.absent(key));
        } else {
            selectors.add(Selector.glob(key, value));
        }
    }

    private static void addExactOrAbsent(List<Selector> selectors, String key, String value) {
        if (value.isEmpty()) {
            selectors.add(Selector.absent(key));
        } else {
            selectors.add(Selector.exact(key, value));
        }
    }
}
