package ru.yandex.solomon.expression.expr.func.util;

import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.compile.SelStatement;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.SelExprFnExternalizeVisitor;
import ru.yandex.solomon.expression.expr.SelExprValue;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValueDouble;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Ivan Tsybulin
 * Returns first timestamp (in seconds) of the time interval
 * for which the code was prepared
 */
@ParametersAreNonnullByDefault
public class SelFnEvalTimeInterval implements SelFuncProvider {

    public static List<SelStatement> patch(Interval interval, List<SelStatement> block) {
        return SelExprFnExternalizeVisitor.replaceFnCall(
            fnCall -> {
                var name = fnCall.getFunc().getName();
                return name.equals("time_interval_end") || name.equals("time_interval_begin");
            },
            fnCall -> {
                var name = fnCall.getFunc().getName();
                long seconds;
                if (name.equals("time_interval_begin")) {
                    seconds = TimeUnit.MILLISECONDS.toSeconds(interval.getBeginMillis());
                } else {
                    seconds = TimeUnit.MILLISECONDS.toSeconds(interval.getEndMillis());
                }
                return new SelExprValue(fnCall.getSourceAst(), new SelValueDouble(seconds));
            },
            block
        );
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("time_interval_begin")
            .help("Returns first timestamp (in seconds) of the time interval " +
                "for which the code was prepared")
            .category(SelFuncCategory.OTHER)
            .returnType(SelTypes.DOUBLE)
            .pure(false)
            .handler(args -> {
                throw new EvaluationException(args.getCallRange(), "should not be call in runtime");
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("time_interval_end")
            .help("Returns last timestamp (in seconds) of the time interval for which the code was prepared")
            .category(SelFuncCategory.OTHER)
            .returnType(SelTypes.DOUBLE)
            .pure(false)
            .handler(args -> {
                throw new EvaluationException(args.getCallRange(), "should not be call in runtime");
            })
            .build());
    }
}
