package ru.yandex.solomon.expression.expr.func.util;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.summary.ImmutableSummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.ImmutableSummaryInt64Snapshot;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.exceptions.TooManyLinesInAggregation;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.ArgsList;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.math.operation.reduce.CombineIterator;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;
import ru.yandex.solomon.model.timeseries.aggregation.collectors.PointValueCollector;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;

import static ru.yandex.solomon.expression.expr.func.SelFuncArgument.arg;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnFallback implements SelFuncProvider {
    private static final String FALLBACK = "fallback";

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
                .category(SelFuncCategory.COMBINE)
                .name(FALLBACK)
                .args(
                    arg("main")
                        .help("Primary expression")
                        .type(SelTypes.GRAPH_DATA_VECTOR),
                    arg("fallback")
                        .help("Fallback expression, used in gapes of primary or as a complete replacement if primary is empty")
                        .type(SelTypes.GRAPH_DATA_VECTOR)
                )
                .returnType(SelTypes.GRAPH_DATA_VECTOR)
                .handler(SelFnFallback::handler)
                .build());
    }

    private static SelValue handler(ArgsList argsList) {
        var mainList = argsList.get(0).castToVector().valueArray();
        var fallbackList = argsList.get(1).castToVector().valueArray();

        if (mainList.length == 0) {
            return argsList.get(1);
        }

        if (fallbackList.length == 0) {
            return argsList.get(0);
        }

        if (mainList.length > 1) {
            throw new TooManyLinesInAggregation(argsList.getRange(0), FALLBACK, Arrays.stream(mainList)
                    .map(SelValue::castToGraphData)
                    .map(v -> v.castToGraphData().getNamedGraphData())
                    .collect(Collectors.toList())
            );
        }

        if (fallbackList.length > 1) {
            throw new TooManyLinesInAggregation(argsList.getRange(1), FALLBACK, Arrays.stream(fallbackList)
                    .map(SelValue::castToGraphData)
                    .map(v -> v.castToGraphData().getNamedGraphData())
                    .collect(Collectors.toList())
            );
        }

        NamedGraphData main = mainList[0].castToGraphData().getNamedGraphData();
        NamedGraphData fallback = fallbackList[0].castToGraphData().getNamedGraphData();

        return new SelValueGraphData(fallbackPoints(main, fallback)).asSingleElementVector();
    }

    private static NamedGraphData fallbackPoints(NamedGraphData mainNamed, NamedGraphData fallbackNamed) {
        AggrGraphDataArrayListOrView main = mainNamed.getAggrGraphDataArrayList();
        AggrGraphDataArrayListOrView fallback = fallbackNamed.getAggrGraphDataArrayList();

        if (main.isEmpty()) {
            return fallbackNamed;
        }

        if (fallback.isEmpty()) {
            return mainNamed;
        }

        MetricType dataType = mainNamed.getDataType();
        AggrGraphDataListIterator fallbackIterator = MetricTypeTransfers.of(
                fallbackNamed.getDataType(),
                dataType,
                fallback.iterator()
        );

        AggrGraphDataListIterator mergedIterator = CombineIterator.of(StockpileColumns.minColumnSet(dataType),
                List.of(main.iterator(), fallbackIterator), new FallbackCollector(dataType));

        return mainNamed.toBuilder()
                .setGraphData(dataType, AggrGraphDataArrayList.of(mergedIterator))
                .build();
    }

    private static class FallbackCollector implements PointValueCollector {
        private final MetricType dataType;
        private AggrPoint point;

        private FallbackCollector(MetricType dataType) {
            this.dataType = dataType;
        }

        @Override
        public void reset() {
            point = null;
        }

        @Override
        public void append(AggrPoint point) {
            if (this.point == null) {
                this.point = point;
                return;
            }

            if (isFalsy(this.point) && !isFalsy(point)) {
                this.point = point;
            }
        }

        @Override
        public boolean compute(AggrPoint point) {
            if (this.point == null || isFalsy(this.point)) {
                return false;
            }
            this.point.copyTo(point);
            return true;
        }

        private boolean isFalsy(AggrPoint point) {
            switch (dataType) {
                case DGAUGE: return isFalsy(point.valueNum);
                case RATE:
                case COUNTER:
                case IGAUGE: return isFalsy(point.longValue);
                case DSUMMARY: return isFalsy(point.summaryDouble);
                case ISUMMARY: return isFalsy(point.summaryInt64);
                case HIST:
                case HIST_RATE: return isFalsy(point.histogram);
                case LOG_HISTOGRAM: return isFalsy(point.logHistogram);
                default:
                    throw new UnsupportedOperationException("Unsupported data type in fallback: " + dataType);
            }
        }

        private static boolean isFalsy(double valueNum) {
            return valueNum == 0 || Double.isNaN(valueNum);
        }

        private static boolean isFalsy(long longValue) {
            return longValue == 0;
        }

        private static boolean isFalsy(@Nullable Histogram histogram) {
            if (histogram == null) {
                return true;
            }
            long total = 0;
            for (int i = 0; i < histogram.count(); i++) {
                total += histogram.value(i);
            }
            return isFalsy(total);
        }

        private static boolean isFalsy(@Nullable LogHistogram logHistogram) {
            if (logHistogram == null) {
                return true;
            }
            long total = logHistogram.getCountZero();
            for (int i = 0; i < logHistogram.countBucket(); i++) {
                total += logHistogram.getBucketValue(i);
            }
            return isFalsy(total);
        }

        private static boolean isFalsy(@Nullable SummaryInt64Snapshot summaryInt64) {
            if (summaryInt64 == null || summaryInt64 == ImmutableSummaryInt64Snapshot.EMPTY) {
                return true;
            }

            return summaryInt64.equals(ImmutableSummaryInt64Snapshot.EMPTY);
        }

        private static boolean isFalsy(@Nullable SummaryDoubleSnapshot summaryDouble) {
            if (summaryDouble == null || summaryDouble == ImmutableSummaryDoubleSnapshot.EMPTY) {
                return true;
            }

            return summaryDouble.equals(ImmutableSummaryDoubleSnapshot.EMPTY);
        }
    }
}
