package ru.yandex.solomon.expression.expr.func.util;

import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.expr.func.SelFuncProvider;
import ru.yandex.solomon.expression.expr.func.SelFuncRegistry;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValueString;

/**
 * Extract label value from a Timeseries. Applicable to a vector of Timeseries.
 * If a label with specified key is missing an empty string is returned. For the
 * vector case, if different values are found an exception is thrown.
 * <p>
 * Example:
 * <pre>let host = get_label(graphData, "host")</pre>
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelFnGetLabel implements SelFuncProvider {

    private static String calculate(NamedGraphData source, String key) {
        var labels = source.getLabels();
        var label = labels.findByKey(key);
        if (label == null) {
            return "";
        }
        return label.getValue();
    }

    @Override
    public void provide(SelFuncRegistry registry) {
        registry.add(SelFunc.newBuilder()
            .name("get_label")
            .help("Extract label value from a Timeseries. Applicable to a vector of Timeseries.\n" +
                "If a label with specified key is missing an empty string is returned. For the " +
                "vector case, if different values are found an exception is thrown.")
            .category(SelFuncCategory.OTHER)
            .args(SelTypes.GRAPH_DATA, SelTypes.STRING)
            .returnType(SelTypes.STRING)
            .handler(args -> {
                var source = args.get(0).castToGraphData().getNamedGraphData();
                var key = args.get(1).castToString().getValue();
                return new SelValueString(calculate(source, key));
            })
            .build());

        registry.add(SelFunc.newBuilder()
            .name("get_label")
            .help("Extract label value from a Timeseries. Applicable to a vector of Timeseries.\n" +
                "If a label with specified key is missing an empty string is returned. For the " +
                "vector case, if different values are found an exception is thrown.")
            .category(SelFuncCategory.OTHER)
            .args(SelTypes.GRAPH_DATA_VECTOR, SelTypes.STRING)
            .returnType(SelTypes.STRING)
            .handler(args -> {
                var source = args.get(0).castToVector().valueArray();
                var key = args.get(1).castToString().getValue();

                Set<String> values = Stream.of(source)
                    .map(value -> calculate(value.castToGraphData().getNamedGraphData(), key))
                    .collect(Collectors.toSet());

                if (values.size() != 1) {
                    throw new EvaluationException(args.getCallRange(), "The label key \"" + key + "\" has different values: " + values);
                }

                return new SelValueString(values.iterator().next());
            })
            .build());
    }
}
